// Dashboard JavaScript
class MetaAdsDashboard {
    constructor() {
        this.apiBase = '/api';
        this.uploadedFiles = [];
        this.sheetsAdCopy = [];
        this.selectedAdAccountId = null; // Will be set to default from .env initially
        this.init();
    }

    init() {
        // Wait for DOM to be fully ready, then initialize
        this.waitForElementsAndInit();
    }

    waitForElementsAndInit(retryCount = 0) {
        const maxRetries = 10;
        const requiredElements = [
            'csv-file-input', 'csv-upload-area', 'load-csv-data', 'download-sample-csv',
            'csv-file-input-duplicate', 'csv-upload-area-duplicate', 'load-csv-data-duplicate', 'download-sample-csv-duplicate',
            'campaign-select', 'campaign-select-duplicate', 'create-ads', 'create-duplicate-adset'
        ];
        
        // Check if all required elements exist
        const missingElements = requiredElements.filter(id => !document.getElementById(id));
        
        const runInit = () => {
            const steps = [
                ['debugElementsExistence', () => this.debugElementsExistence()],
                ['setupEventListeners',    () => this.setupEventListeners()],
                ['setupTabNavigation',     () => this.setupTabNavigation()],
                ['loadInitialData',        () => this.loadInitialData()],
                ['startDataRefresh',       () => this.startDataRefresh()],
            ];
            for (const [name, fn] of steps) {
                try {
                    fn();
                    console.log(`✅ ${name} OK`);
                } catch(e) {
                    console.error(`❌ CRASHED in ${name}:`, e);
                }
            }
        };

        if (missingElements.length === 0) {
            console.log('✅ All elements found, initializing...');
            runInit();
        } else if (retryCount < maxRetries) {
            console.log(`⏳ Waiting for elements (${retryCount + 1}/${maxRetries}). Missing:`, missingElements);
            setTimeout(() => this.waitForElementsAndInit(retryCount + 1), 100);
        } else {
            console.error('❌ Missing elements:', missingElements, '— initializing anyway');
            runInit();
        }
    }

    debugElementsExistence() {
        const requiredElements = [
            'csv-file-input', 'csv-upload-area', 'load-csv-data', 'download-sample-csv',
            'csv-file-input-duplicate', 'csv-upload-area-duplicate', 'load-csv-data-duplicate', 'download-sample-csv-duplicate',
            'campaign-select', 'campaign-select-duplicate', 'create-ads', 'create-duplicate-adset'
        ];
        
        console.log('🔍 Checking for required CSV elements:');
        requiredElements.forEach(id => {
            const element = document.getElementById(id);
            console.log(`  ${id}: ${element ? '✅ Found' : '❌ Missing'}`);
        });
    }

    setupEventListeners() {
        // Tab navigation
        this.setupTabNavigation();
        
        
        // Ad Account Selection
        document.getElementById('ad-account-select').addEventListener('change', (e) => {
            this.selectedAdAccountId = e.target.value;
            console.log('🏦 Ad account changed to:', this.selectedAdAccountId);
            this.loadCampaigns(); // Reload campaigns for the selected ad account
        });
        document.getElementById('ad-account-select-duplicate').addEventListener('change', (e) => {
            this.selectedAdAccountId = e.target.value;
            console.log('🏦 Ad account changed to:', this.selectedAdAccountId);
            this.loadCampaigns(); // Reload campaigns for the selected ad account (populates both dropdowns)
        });

        // Creative Upload & Ad Creation Tab - Existing AdSet
        document.getElementById('campaign-select').addEventListener('change', (e) => this.loadAdSets(e.target.value));
        document.getElementById('adset-select').addEventListener('change', (e) => this.loadReferenceAds(e.target.value));

        // Creative Upload & Ad Creation Tab - Duplicate AdSet
        document.getElementById('campaign-select-duplicate').addEventListener('change', (e) => this.loadAdSetsDuplicate(e.target.value));
        document.getElementById('adset-select-duplicate').addEventListener('change', (e) => this.loadReferenceAdsDuplicate(e.target.value));
        
        // CSV upload - with better error handling
        const loadCsvBtn = document.getElementById('load-csv-data');
        const downloadCsvBtn = document.getElementById('download-sample-csv');
        const loadCsvBtnDuplicate = document.getElementById('load-csv-data-duplicate');
        const downloadCsvBtnDuplicate = document.getElementById('download-sample-csv-duplicate');
        
        if (loadCsvBtn) {
            loadCsvBtn.addEventListener('click', () => {
                console.log('📄 Upload CSV button clicked');
                this.loadCSVData();
            });
        } else {
            console.error('❌ load-csv-data button not found');
        }
        
        if (downloadCsvBtn) {
            downloadCsvBtn.addEventListener('click', () => this.downloadSampleCSV());
        } else {
            console.error('❌ download-sample-csv button not found');
        }

        // Duplicate tab CSV upload
        if (loadCsvBtnDuplicate) {
            loadCsvBtnDuplicate.addEventListener('click', () => {
                console.log('📄 Upload CSV button clicked (duplicate)');
                this.loadCSVDataDuplicate();
            });
        } else {
            console.error('❌ load-csv-data-duplicate button not found');
        }
        
        if (downloadCsvBtnDuplicate) {
            downloadCsvBtnDuplicate.addEventListener('click', () => this.downloadSampleCSV());
        } else {
            console.error('❌ download-sample-csv-duplicate button not found');
        }
        
        document.getElementById('create-ads').addEventListener('click', () => this.createAdsFromUploads());
        document.getElementById('create-duplicate-adset').addEventListener('click', () => this.createDuplicateAdSet());

        // File upload - existing adset
        const uploadArea = document.getElementById('upload-area');
        const fileInput = document.getElementById('file-input');
        
        // File upload - duplicate adset
        const uploadAreaDuplicate = document.getElementById('upload-area-duplicate');
        const fileInputDuplicate = document.getElementById('file-input-duplicate');
        
        uploadArea.addEventListener('click', () => fileInput.click());
        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.classList.add('dragover');
        });
        uploadArea.addEventListener('dragleave', () => {
            uploadArea.classList.remove('dragover');
        });
        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');
            this.handleFileUpload(e.dataTransfer.files);
        });
        
        fileInput.addEventListener('change', (e) => {
            this.handleFileUpload(e.target.files);
        });

        // Duplicate file upload area event listeners
        if (uploadAreaDuplicate && fileInputDuplicate) {
            uploadAreaDuplicate.addEventListener('click', () => fileInputDuplicate.click());
            uploadAreaDuplicate.addEventListener('dragover', (e) => {
                e.preventDefault();
                uploadAreaDuplicate.classList.add('dragover');
            });
            uploadAreaDuplicate.addEventListener('dragleave', () => {
                uploadAreaDuplicate.classList.remove('dragover');
            });
            uploadAreaDuplicate.addEventListener('drop', (e) => {
                e.preventDefault();
                uploadAreaDuplicate.classList.remove('dragover');
                this.handleFileUpload(e.dataTransfer.files);
            });
            
            fileInputDuplicate.addEventListener('change', (e) => {
                this.handleFileUpload(e.target.files);
            });
        }

        // CSV upload
        const csvUploadArea = document.getElementById('csv-upload-area');
        const csvFileInput = document.getElementById('csv-file-input');
        
        if (!csvUploadArea || !csvFileInput) {
            console.error('❌ CSV upload elements not found:', {
                csvUploadArea: !!csvUploadArea,
                csvFileInput: !!csvFileInput
            });
            return;
        }
        
        csvUploadArea.addEventListener('click', () => {
            const currentInput = document.getElementById('csv-file-input');
            if (currentInput) {
                currentInput.click();
            } else {
                console.error('❌ CSV file input not found when clicking upload area');
            }
        });
        csvUploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            csvUploadArea.classList.add('dragover');
        });
        csvUploadArea.addEventListener('dragleave', () => {
            csvUploadArea.classList.remove('dragover');
        });
        csvUploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            csvUploadArea.classList.remove('dragover');
            if (e.dataTransfer.files[0] && e.dataTransfer.files[0].name.endsWith('.csv')) {
                csvFileInput.files = e.dataTransfer.files;
                this.handleCSVFileSelect(e.dataTransfer.files[0]);
            } else {
                this.showAlert('Please drop a CSV file', 'error');
            }
        });
        
        csvFileInput.addEventListener('change', (e) => {
            console.log('📄 CSV file input changed, files:', e.target.files.length);
            if (e.target.files[0]) {
                console.log('📄 File selected:', e.target.files[0].name);
                this.handleCSVFileSelect(e.target.files[0]);
            } else {
                console.log('📄 No file selected');
            }
        });

        // CSV upload for duplicate tab
        const csvUploadAreaDuplicate = document.getElementById('csv-upload-area-duplicate');
        const csvFileInputDuplicate = document.getElementById('csv-file-input-duplicate');
        
        if (csvUploadAreaDuplicate && csvFileInputDuplicate) {
            csvUploadAreaDuplicate.addEventListener('click', () => {
                const currentInput = document.getElementById('csv-file-input-duplicate');
                if (currentInput) {
                    currentInput.click();
                } else {
                    console.error('❌ CSV file input duplicate not found when clicking upload area');
                }
            });
            csvUploadAreaDuplicate.addEventListener('dragover', (e) => {
                e.preventDefault();
                csvUploadAreaDuplicate.classList.add('dragover');
            });
            csvUploadAreaDuplicate.addEventListener('dragleave', () => {
                csvUploadAreaDuplicate.classList.remove('dragover');
            });
            csvUploadAreaDuplicate.addEventListener('drop', (e) => {
                e.preventDefault();
                csvUploadAreaDuplicate.classList.remove('dragover');
                if (e.dataTransfer.files[0] && e.dataTransfer.files[0].name.endsWith('.csv')) {
                    csvFileInputDuplicate.files = e.dataTransfer.files;
                    this.handleCSVFileSelectDuplicate(e.dataTransfer.files[0]);
                } else {
                    this.showAlert('Please drop a CSV file', 'error');
                }
            });
            
            csvFileInputDuplicate.addEventListener('change', (e) => {
                console.log('📄 CSV file input duplicate changed, files:', e.target.files.length);
                if (e.target.files[0]) {
                    console.log('📄 File selected (duplicate):', e.target.files[0].name);
                    this.handleCSVFileSelectDuplicate(e.target.files[0]);
                } else {
                    console.log('❌ No file selected (duplicate)');
                }
            });
        }

        // Performance Monitoring removed
    }

    setupTabNavigation() {
        console.log('🚀 Setting up tab navigation...');
        
        // Main tabs - find tabs that are direct children of .card > .tabs
        const mainTabsContainer = document.querySelector('.container > .card > .tabs');
        const mainTabs = mainTabsContainer ? mainTabsContainer.querySelectorAll('.tab[data-tab]') : [];
        console.log('📑 Found main tabs:', mainTabs.length);
        
        if (mainTabs.length === 0) {
            console.error('❌ No main tabs found! Check HTML structure.');
        }
        
        mainTabs.forEach(tab => {
            tab.addEventListener('click', () => {
                const targetTab = tab.dataset.tab;
                
                // Remove active class from all main tabs and contents
                mainTabs.forEach(t => t.classList.remove('active'));
                document.querySelectorAll('.container > .card > .tab-content').forEach(content => {
                    content.classList.remove('active');
                });
                
                // Add active class to clicked tab and corresponding content
                tab.classList.add('active');
                const targetContent = document.getElementById(`${targetTab}-tab`);
                if (targetContent) {
                    targetContent.classList.add('active');
                }
                
                // Load tab-specific data
                this.loadTabData(targetTab);
            });
        });

        // Sub-tabs for Creative Upload & Ad Creation
        const creativeSubTabs = document.querySelectorAll('#creative-ads-tab .tabs .tab');
        creativeSubTabs.forEach(tab => {
            tab.addEventListener('click', () => {
                const targetTab = tab.dataset.tab;
                
                // Remove active class from sub-tabs and contents
                creativeSubTabs.forEach(t => t.classList.remove('active'));
                document.querySelectorAll('#creative-ads-tab .tab-content').forEach(content => {
                    content.classList.remove('active');
                });
                
                // Add active class
                tab.classList.add('active');
                const targetContent = document.getElementById(`${targetTab}-tab`);
                if (targetContent) {
                    targetContent.classList.add('active');
                }
            });
        });

        // Performance monitoring sub-tabs will be initialized separately
        console.log('✅ Tab navigation setup complete');
    }

    async loadInitialData() {
        // Load ad accounts first, then campaigns
        await this.loadAdAccounts();
        await Promise.all([
            this.loadCampaigns()
            // Disabled monitoring status to prevent errors
            // this.loadMonitoringStatus()
        ]);
    }

    startDataRefresh() {
        // Refresh data every 30 seconds
        setInterval(() => {
            // Disabled monitoring status to prevent errors
            // this.loadMonitoringStatus();
            if (false) {
                // this.loadUnderperformingAds();
            }
        }, 30000);
    }

    async loadReferenceAds(adsetId) {
        console.log('📋 Loading reference ads for adset:', adsetId);
        const referenceAdSelect = document.getElementById('reference-ad-select');
        
        if (!adsetId) {
            referenceAdSelect.innerHTML = '<option value="">Select an adset first</option>';
            referenceAdSelect.disabled = true;
            return;
        }
        
        try {
            referenceAdSelect.innerHTML = '<option value="">Loading ads...</option>';
            referenceAdSelect.disabled = false;
            
            const response = await fetch(`${this.apiBase}/campaigns/adset/${adsetId}/ads`);
            console.log('📊 Reference ads API Response status:', response.status);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const data = await response.json();
            console.log('📊 Reference ads API Response data:', data);
            
            referenceAdSelect.innerHTML = '<option value="">Select a reference ad...</option>';
            
            if (data.ads && data.ads.length > 0) {
                data.ads.forEach((ad, index) => {
                    const option = document.createElement('option');
                    option.value = ad.id;
                    const status = ad.effective_status || ad.status || 'UNKNOWN';
                    option.textContent = `${ad.name} (${status})`;
                    referenceAdSelect.appendChild(option);
                    console.log(`➕ Added reference ad ${index + 1}: ${ad.name}`);
                });
                console.log(`✅ Successfully loaded ${data.ads.length} reference ads`);
            } else {
                referenceAdSelect.innerHTML += '<option value="" disabled>No ads found</option>';
                console.warn('⚠️ No ads found for adset');
            }
            
            this.validateCreateAdsButton();
            
        } catch (error) {
            console.error('❌ Error loading reference ads:', error);
            referenceAdSelect.innerHTML = '<option value="" disabled>Error loading ads</option>';
            referenceAdSelect.disabled = true;
            this.showAlert('Error loading reference ads: ' + error.message, 'error');
        }
    }

    async loadAdSets(campaignId) {
        console.log('📋 Loading adsets for campaign:', campaignId);
        const adsetSelect = document.getElementById('adset-select');
        const referenceAdSelect = document.getElementById('reference-ad-select');
        
        if (!campaignId) {
            adsetSelect.innerHTML = '<option value="">Select a campaign first</option>';
            adsetSelect.disabled = true;
            referenceAdSelect.innerHTML = '<option value="">Select an adset first</option>';
            referenceAdSelect.disabled = true;
            return;
        }
        
        try {
            adsetSelect.innerHTML = '<option value="">Loading adsets...</option>';
            adsetSelect.disabled = false;
            
            const response = await fetch(`${this.apiBase}/campaigns/${campaignId}`);
            console.log('📊 Adsets API Response status:', response.status);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const data = await response.json();
            console.log('📊 Adsets API Response data:', data);
            
            adsetSelect.innerHTML = '<option value="">Select an adset...</option>';
            
            if (data.adsets && data.adsets.length > 0) {
                data.adsets.forEach((adset, index) => {
                    const option = document.createElement('option');
                    option.value = adset.id;
                    const budget = adset.daily_budget ? `$${(adset.daily_budget / 100).toFixed(2)}/day` : 'No budget';
                    option.textContent = `${adset.name} (${budget})`;
                    adsetSelect.appendChild(option);
                    console.log(`➕ Added adset ${index + 1}: ${adset.name}`);
                });
                console.log(`✅ Successfully loaded ${data.adsets.length} adsets`);
            } else {
                adsetSelect.innerHTML += '<option value="" disabled>No adsets found</option>';
                console.warn('⚠️ No adsets found for campaign');
            }
            
        } catch (error) {
            console.error('❌ Error loading adsets:', error);
            adsetSelect.innerHTML = '<option value="" disabled>Error loading adsets</option>';
            adsetSelect.disabled = true;
            this.showAlert('Error loading adsets: ' + error.message, 'error');
        }
    }

    async loadTabData(tabName) {
        if (tabName === 'creative-ads') await this.loadCampaigns();
    }


    async loadAdAccounts() {
        const adAccountSelect = document.getElementById('ad-account-select');
        const adAccountSelectDuplicate = document.getElementById('ad-account-select-duplicate');

        const setAccount = (select, id) => {
            if (!select) return;
            select.innerHTML = '';
            const option = document.createElement('option');
            option.value = id;
            option.textContent = id;
            option.selected = true;
            select.appendChild(option);
        };

        try {
            const response = await fetch('/api/campaigns/ad-accounts');
            const data = await response.json();
            const account = (data.adAccounts || [])[0];
            if (account) {
                setAccount(adAccountSelect, account.id);
                setAccount(adAccountSelectDuplicate, account.id);
                this.selectedAdAccountId = account.id;
                console.log('✅ Ad account loaded:', account.id);
                this.loadCampaigns();
            }
        } catch (error) {
            console.error('❌ loadAdAccounts failed:', error);
            if (adAccountSelect) adAccountSelect.innerHTML = '<option value="">Error loading</option>';
            if (adAccountSelectDuplicate) adAccountSelectDuplicate.innerHTML = '<option value="">Error loading</option>';
        }
    }


    async loadCampaigns() {
        console.log('📋 Loading campaigns...');
        try {
            const campaignSelect = document.getElementById('campaign-select');
            if (!campaignSelect) {
                throw new Error('Campaign select element not found');
            }

            campaignSelect.innerHTML = '<option value="">Loading campaigns...</option>';

            // Send selected ad account as query parameter
            const queryParams = this.selectedAdAccountId ? `?adAccountId=${encodeURIComponent(this.selectedAdAccountId)}` : '';
            const response = await fetch(`${this.apiBase}/campaigns${queryParams}`);
            console.log('📊 API Response status:', response.status);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const data = await response.json();
            console.log('📊 API Response data:', data);
            
            campaignSelect.innerHTML = '<option value="">Select a campaign...</option>';
            
            if (!data.campaigns || data.campaigns.length === 0) {
                campaignSelect.innerHTML += '<option value="" disabled>No campaigns found</option>';
                console.warn('⚠️ No campaigns found in API response');
                return;
            }
            
            data.campaigns.forEach((campaign, index) => {
                const option = document.createElement('option');
                option.value = campaign.id;
                const status = campaign.effective_status || campaign.status || 'UNKNOWN';
                option.textContent = `${campaign.name} (${status})`;
                campaignSelect.appendChild(option);
                console.log(`➕ Added campaign ${index + 1}: ${campaign.name}`);
            });
            
            console.log(`✅ Successfully loaded ${data.campaigns.length} campaigns`);
            
        } catch (error) {
            console.error('❌ Error loading campaigns:', error);
            const campaignSelect = document.getElementById('campaign-select');
            if (campaignSelect) {
                campaignSelect.innerHTML = '<option value="" disabled>Error loading campaigns</option>';
            }
            this.showAlert('Error loading campaigns: ' + error.message, 'error');
        }
    }

    async handleFileUpload(files) {
        this.uploadedFiles = Array.from(files);
        
        const filesListElement = document.getElementById('files-list');
        const uploadedFilesDiv = document.getElementById('uploaded-files');
        
        if (files.length === 0) {
            uploadedFilesDiv.classList.add('hidden');
            return;
        }
        
        uploadedFilesDiv.classList.remove('hidden');
        
        const filesHtml = Array.from(files).map(file => `
            <div style="padding: 0.5rem; background: #f7fafc; border: 1px solid #e2e8f0; border-radius: 4px; margin: 0.25rem 0; display: flex; justify-content: space-between; align-items: center;">
                <span style="font-size: 0.875rem;">${file.name}</span>
                <span style="font-size: 0.75rem; color: #718096;">${(file.size / 1024 / 1024).toFixed(2)} MB</span>
            </div>
        `).join('');
        
        filesListElement.innerHTML = filesHtml;
        
        this.validateCreateAdsButton();
        
        this.showAlert(`${files.length} files ready for upload`, 'success');
    }

    async handleCSVFileSelect(file) {
        console.log('📄 handleCSVFileSelect called with file:', file.name, file.size);
        const csvUploadArea = document.getElementById('csv-upload-area');
        if (!csvUploadArea) {
            console.error('❌ CSV upload area element not found');
            return;
        }
        
        // Find and preserve the file input element
        const fileInput = csvUploadArea.querySelector('#csv-file-input');
        
        // Update only the visible content, keep the file input
        const paragraphs = csvUploadArea.querySelectorAll('p');
        paragraphs.forEach(p => p.remove());
        
        // Add new content
        const fileName = document.createElement('p');
        fileName.textContent = `📄 ${file.name}`;
        
        const fileSize = document.createElement('p');
        fileSize.style.fontSize = '0.875rem';
        fileSize.style.color = '#718096';
        fileSize.style.marginTop = '0.5rem';
        fileSize.textContent = `${(file.size / 1024).toFixed(1)} KB - Ready to upload`;
        
        // Insert new content before the file input
        if (fileInput) {
            csvUploadArea.insertBefore(fileName, fileInput);
            csvUploadArea.insertBefore(fileSize, fileInput);
        } else {
            csvUploadArea.appendChild(fileName);
            csvUploadArea.appendChild(fileSize);
        }
        
        console.log('📄 Updated CSV upload area display, preserved file input');
    }

    async loadCSVData() {
        const csvFileInput = document.getElementById('csv-file-input');
        
        if (!csvFileInput) {
            this.showAlert('Please select a CSV file first', 'error');
            return;
        }
        
        if (!csvFileInput.files[0]) {
            this.showAlert('Please select a CSV file first', 'error');
            return;
        }
        
        try {
            const button = document.getElementById('load-csv-data');
            button.innerHTML = '<span class="loading"></span> Processing...';
            button.disabled = true;
            
            const formData = new FormData();
            formData.append('csvFile', csvFileInput.files[0]);
            
            const response = await fetch(`${this.apiBase}/csv/upload-ad-copy`, {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            if (data.success && data.adCopy && data.adCopy.length > 0) {
                this.sheetsAdCopy = data.adCopy;
                
                const previewHtml = data.adCopy.slice(0, 3).map((copy, index) => `
                    <div style="margin: 0.5rem 0; padding: 0.5rem; background: white; border: 1px solid #e2e8f0; border-radius: 4px;">
                        <div style="font-weight: 500; font-size: 0.875rem;">Row ${index + 1}: ${copy.bookId} - ${copy.variation}</div>
                        <div style="font-size: 0.8rem; color: #4a5568;">Primary: ${copy.primaryText || 'N/A'}</div>
                        <div style="font-size: 0.8rem; color: #4a5568;">Headline: ${copy.headline || 'N/A'}</div>
                        <div style="font-size: 0.8rem; color: #4a5568;">Description: ${copy.description || 'N/A'}</div>
                        <div style="font-size: 0.75rem; color: #718096;">Landing: ${copy.landingPageUrl || 'N/A'}</div>
                    </div>
                `).join('');
                
                const moreText = data.adCopy.length > 3 ? `<div style="font-size: 0.8rem; font-style: italic; margin-top: 0.5rem;">...and ${data.adCopy.length - 3} more rows</div>` : '';
                
                document.getElementById('csv-data-preview').innerHTML = previewHtml + moreText;
                document.getElementById('csv-preview').classList.remove('hidden');
                
                this.showAlert(data.message || `Loaded ${data.adCopy.length} ad copy variations`, 'success');
            } else {
                throw new Error(data.error || 'No ad copy data found in CSV');
            }
            
            button.textContent = 'Upload CSV';
            button.disabled = false;
            
            this.validateCreateAdsButton();
            
        } catch (error) {
            console.error('Error loading CSV data:', error);
            this.showAlert('Error processing CSV file: ' + error.message, 'error');
            
            const button = document.getElementById('load-csv-data');
            button.textContent = 'Upload CSV';
            button.disabled = false;
        }
    }

    async downloadSampleCSV() {
        try {
            const response = await fetch(`${this.apiBase}/csv/sample-format`);
            const blob = await response.blob();
            
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'ad-copy-sample.csv';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            window.URL.revokeObjectURL(url);
            
            this.showAlert('Sample CSV downloaded', 'success');
        } catch (error) {
            console.error('Error downloading sample CSV:', error);
            this.showAlert('Error downloading sample CSV: ' + error.message, 'error');
        }
    }

    validateCreateAdsButton() {
        const createButton = document.getElementById('create-ads');
        const hasFiles = this.uploadedFiles && this.uploadedFiles.length > 0;
        const hasAdCopy = this.sheetsAdCopy && this.sheetsAdCopy.length > 0;
        const hasAdSet = document.getElementById('adset-select').value;
        const hasReferenceAd = document.getElementById('reference-ad-select').value;
        
        if (hasFiles && hasAdCopy && hasAdSet && hasReferenceAd) {
            createButton.disabled = false;
            createButton.textContent = `Create ${this.uploadedFiles.length * this.sheetsAdCopy.length} Ad Combinations`;
        } else {
            createButton.disabled = true;
            createButton.textContent = 'Create All Ad Combinations';
        }
    }

    async createAdsFromUploads() {
        console.log('🚀 createAdsFromUploads() called');
        
        const adsetId = document.getElementById('adset-select').value;
        const referenceAdId = document.getElementById('reference-ad-select').value;
        
        console.log('📋 Current state:', {
            uploadedFiles: this.uploadedFiles?.length || 0,
            csvAdCopy: this.sheetsAdCopy?.length || 0,
            adsetId,
            referenceAdId
        });
        
        if (!this.uploadedFiles || this.uploadedFiles.length === 0) {
            console.error('❌ No uploaded files');
            this.showAlert('Please upload creative files first', 'error');
            return;
        }
        
        if (!this.sheetsAdCopy || this.sheetsAdCopy.length === 0) {
            console.error('❌ No CSV ad copy data');
            this.showAlert('Please upload CSV ad copy data first', 'error');
            return;
        }
        
        if (!adsetId || !referenceAdId) {
            console.error('❌ Missing adset or reference ad selection:', { adsetId, referenceAdId });
            this.showAlert('Please select campaign, adset, and reference ad', 'error');
            return;
        }
        
        console.log('✅ All validation passed, proceeding with ad creation');
        
        try {
            const button = document.getElementById('create-ads');
            const originalText = button.textContent;
            button.innerHTML = '<span class="loading"></span> Creating Ads...';
            button.disabled = true;
            
            // First upload the creative files
            console.log('📤 Uploading creative files to server...');
            const formData = new FormData();
            for (let file of this.uploadedFiles) {
                formData.append('creatives', file);
                console.log(`  - Adding file: ${file.name} (${file.size} bytes)`);
            }
            formData.append('adsetId', adsetId);
            formData.append('adAccountId', this.selectedAdAccountId);  // Send selected ad account
            
            console.log(`📡 Calling: ${this.apiBase}/creatives/upload-for-adset`);
            const uploadResponse = await fetch(`${this.apiBase}/creatives/upload-for-adset`, {
                method: 'POST',
                body: formData
            });
            
            console.log('📊 Upload response status:', uploadResponse.status);
            const uploadData = await uploadResponse.json();
            console.log('📊 Upload response data:', uploadData);
            
            if (!uploadData.success) {
                throw new Error('Failed to upload creative files');
            }

            // Create filename mapping: creativeId -> originalName
            const creativeFilenames = {};
            if (uploadData.files) {
                uploadData.files.forEach(file => {
                    const creativeId = file.metaHash || file.metaVideoId;
                    if (creativeId && file.originalName) {
                        creativeFilenames[creativeId] = file.originalName;
                    }
                });
            }
            console.log('📁 Creative filename mapping:', creativeFilenames);

            // Then create ads with the uploaded creatives and ad copy
            console.log('🎯 Creating ads with uploaded creatives and CSV ad copy...');
            console.log('📋 Payload for ad creation:', {
                adsetId,
                referenceAdId,
                creativeIds: uploadData.creativeIds,
                adCopyVariations: this.sheetsAdCopy,
                creativeFilenames
            });

            console.log(`📡 Calling: ${this.apiBase}/campaigns/create-ads-batch`);
            const createResponse = await fetch(`${this.apiBase}/campaigns/create-ads-batch`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    adsetId,
                    referenceAdId,
                    creativeIds: uploadData.creativeIds,
                    adCopyVariations: this.sheetsAdCopy,
                    creativeFilenames,
                    adAccountId: this.selectedAdAccountId  // Send selected ad account
                })
            });
            
            console.log('📊 Create ads response status:', createResponse.status);
            const createData = await createResponse.json();
            console.log('📊 Create ads response data:', createData);
            
            this.showAlert(
                `Campaign creation completed: ${createData.successful} ads created, ${createData.failed} failed`,
                createData.failed > 0 ? 'warning' : 'success'
            );
            
            // Reset form
            this.uploadedFiles = [];
            this.sheetsAdCopy = [];
            document.getElementById('uploaded-files').classList.add('hidden');
            document.getElementById('csv-preview').classList.add('hidden');
            document.getElementById('file-input').value = '';
            document.getElementById('csv-file-input').value = '';
            
            // Reset CSV upload area
            const csvUploadArea = document.getElementById('csv-upload-area');
            csvUploadArea.innerHTML = `
                <p>📄 Drop CSV file here or click to select</p>
                <p style="font-size: 0.875rem; color: #718096; margin-top: 0.5rem;">
                    Format: BookID, Variation, PrimaryText, Headline, Description
                </p>
            `;
            
            button.textContent = originalText;
            button.disabled = true;
            
        } catch (error) {
            console.error('Error creating ads:', error);
            this.showAlert('Error creating ads: ' + error.message, 'error');
            
            const button = document.getElementById('create-ads');
            button.textContent = 'Create All Ad Combinations';
            button.disabled = false;
        }
    }

    showAlert(message, type) {
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type}`;
        alertDiv.textContent = message;
        
        const container = document.querySelector('.container');
        container.insertBefore(alertDiv, container.firstChild);
        
        setTimeout(() => {
            alertDiv.remove();
        }, 5000);
    }

    formatTime(date) {
        return date.toLocaleTimeString('en-US', {
            hour: '2-digit',
            minute: '2-digit',
            second: '2-digit'
        });
    }


    // KPI Monitoring Methods
    async createAdsFromUploads() {
        console.log('🚀 createAdsFromUploads() called');
        
        try {
            // Get selected values
            const adsetId = document.getElementById('adset-select').value;
            const referenceAdId = document.getElementById('reference-ad-select').value;
            
            if (!adsetId || !referenceAdId) {
                this.showAlert('Please select both an AdSet and Reference Ad', 'error');
                return;
            }
            
            if (this.uploadedFiles.length === 0) {
                this.showAlert('Please upload creative files first', 'error');
                return;
            }
            
            if (this.sheetsAdCopy.length === 0) {
                this.showAlert('Please upload CSV ad copy data first', 'error');
                return;
            }
            
            const button = document.getElementById('create-ads');
            const originalText = button.innerHTML;
            button.innerHTML = '⏳ Creating ads...';
            button.disabled = true;
            
            // Step 1: Upload creative files to server and get Meta hashes
            console.log('📤 Uploading creative files to server...');
            const formData = new FormData();
            this.uploadedFiles.forEach(file => {
                formData.append('creatives', file);
            });
            formData.append('adsetId', adsetId);
            formData.append('adAccountId', this.selectedAdAccountId);  // Send selected ad account

            const uploadResponse = await fetch(`${this.apiBase}/creatives/upload-for-adset`, {
                method: 'POST',
                body: formData
            });
            
            const uploadResult = await uploadResponse.json();
            
            if (!uploadResult.success || uploadResult.creativeIds.length === 0) {
                throw new Error('Failed to upload creative files: ' + (uploadResult.error || 'No creative IDs returned'));
            }
            
            console.log('✅ Creative files uploaded successfully. Meta hashes:', uploadResult.creativeIds);

            // Create filename mapping: creativeId -> originalName
            const creativeFilenames = {};
            if (uploadResult.files) {
                uploadResult.files.forEach(file => {
                    const creativeId = file.metaHash || file.metaVideoId;
                    if (creativeId && file.originalName) {
                        creativeFilenames[creativeId] = file.originalName;
                    }
                });
            }
            console.log('📁 Creative filename mapping:', creativeFilenames);

            // Step 2: Create ads with uploaded creatives and CSV ad copy
            console.log('🎯 Creating ads with uploaded creatives and CSV ad copy...');
            const createResponse = await fetch(`${this.apiBase}/campaigns/create-ads-batch`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    adsetId: adsetId,
                    referenceAdId: referenceAdId,
                    creativeIds: uploadResult.creativeIds, // Meta image hashes
                    adCopyVariations: this.sheetsAdCopy,
                    creativeFilenames,
                    adAccountId: this.selectedAdAccountId  // Send selected ad account
                })
            });
            
            const createResult = await createResponse.json();
            
            if (!createResult.success) {
                throw new Error('Failed to create ads: ' + (createResult.error || 'Unknown error'));
            }
            
            // Success! Show results
            const totalAds = createResult.successful || 0;
            const failedAds = createResult.failed || 0;
            
            let message = `✅ Successfully created ${totalAds} ads`;
            if (failedAds > 0) {
                message += ` (${failedAds} failed)`;
            }
            
            this.showAlert(message, 'success');
            
            console.log('🎉 Ad creation completed:', {
                successful: totalAds,
                failed: failedAds,
                results: createResult.results
            });
            
        } catch (error) {
            console.error('❌ Error creating ads:', error);
            this.showAlert('Error creating ads: ' + error.message, 'error');
        } finally {
            const button = document.getElementById('create-ads');
            button.innerHTML = 'Create All Ad Combinations';
            button.disabled = false;
        }
    }

    // Campaign and adset loading functions
    async loadInitialData() {
        await this.loadCampaigns();
    }

    async loadCampaigns() {
        try {
            // Send selected ad account as query parameter
            const queryParams = this.selectedAdAccountId ? `?adAccountId=${encodeURIComponent(this.selectedAdAccountId)}` : '';
            const response = await fetch(`${this.apiBase}/campaigns${queryParams}`);
            const data = await response.json();

            // Populate both campaign dropdowns
            const campaignSelect = document.getElementById('campaign-select');
            const campaignSelectDuplicate = document.getElementById('campaign-select-duplicate');
            
            if (campaignSelect) {
                campaignSelect.innerHTML = '<option value="">Select a campaign...</option>';
                data.campaigns.forEach(campaign => {
                    const option = document.createElement('option');
                    option.value = campaign.id;
                    option.textContent = `${campaign.name} (${campaign.effective_status})`;
                    campaignSelect.appendChild(option);
                });
            }
            
            if (campaignSelectDuplicate) {
                campaignSelectDuplicate.innerHTML = '<option value="">Select a campaign...</option>';
                data.campaigns.forEach(campaign => {
                    const option = document.createElement('option');
                    option.value = campaign.id;
                    option.textContent = `${campaign.name} (${campaign.effective_status})`;
                    campaignSelectDuplicate.appendChild(option);
                });
            }
            
        } catch (error) {
            console.error('Error loading campaigns:', error);
            this.showAlert('Error loading campaigns: ' + error.message, 'error');
        }
    }

    async loadAdSets(campaignId) {
        if (!campaignId) {
            document.getElementById('adset-select').innerHTML = '<option value="">Select a campaign first</option>';
            document.getElementById('adset-select').disabled = true;
            document.getElementById('reference-ad-select').innerHTML = '<option value="">Select an adset first</option>';
            document.getElementById('reference-ad-select').disabled = true;
            return;
        }

        try {
            const response = await fetch(`${this.apiBase}/campaigns/${campaignId}`);
            const data = await response.json();
            
            const adsetSelect = document.getElementById('adset-select');
            adsetSelect.innerHTML = '<option value="">Select an adset...</option>';
            
            data.adsets.forEach(adset => {
                const option = document.createElement('option');
                option.value = adset.id;
                option.textContent = `${adset.name} (${adset.effective_status})`;
                adsetSelect.appendChild(option);
            });
            
            adsetSelect.disabled = false;
            
        } catch (error) {
            console.error('Error loading adsets:', error);
            this.showAlert('Error loading adsets: ' + error.message, 'error');
        }
    }

    async loadReferenceAds(adsetId) {
        if (!adsetId) {
            document.getElementById('reference-ad-select').innerHTML = '<option value="">Select an adset first</option>';
            document.getElementById('reference-ad-select').disabled = true;
            return;
        }

        try {
            const response = await fetch(`${this.apiBase}/campaigns/adset/${adsetId}/ads`);
            const data = await response.json();
            
            const referenceAdSelect = document.getElementById('reference-ad-select');
            referenceAdSelect.innerHTML = '<option value="">Select a reference ad...</option>';
            
            data.ads.forEach(ad => {
                const option = document.createElement('option');
                option.value = ad.id;
                option.textContent = `${ad.name} (${ad.effective_status})`;
                referenceAdSelect.appendChild(option);
            });
            
            referenceAdSelect.disabled = false;
            
        } catch (error) {
            console.error('Error loading reference ads:', error);
            this.showAlert('Error loading reference ads: ' + error.message, 'error');
        }
    }

    // CSV handling functions (original tab) - using API route
    loadCSVData() {
        console.log('📄 loadCSVData() called');
        const fileInput = document.getElementById('csv-file-input');
        if (fileInput && fileInput.files.length > 0) {
            this.uploadCSVFile(fileInput.files[0], 'original');
        } else {
            this.showAlert('Please select a CSV file first', 'error');
        }
    }

    handleCSVFileSelect(file) {
        console.log('📄 handleCSVFileSelect() called with file:', file.name);
        
        const csvUploadArea = document.getElementById('csv-upload-area');
        const fileInput = csvUploadArea.querySelector('#csv-file-input');
        
        // Update upload area display - preserve the file input element
        const paragraphs = csvUploadArea.querySelectorAll('p');
        paragraphs.forEach(p => p.remove());
        
        const newParagraph = document.createElement('p');
        newParagraph.innerHTML = `📄 Selected: <strong>${file.name}</strong> (${(file.size / 1024).toFixed(1)} KB)`;
        csvUploadArea.insertBefore(newParagraph, fileInput);
        
        // Store the file for later upload when user clicks the button
        console.log('📄 File stored, ready for upload when user clicks Upload CSV button');
    }

    async uploadCSVFile(file, tabType = 'original') {
        try {
            console.log(`📤 Uploading CSV file to server: ${file.name} (${tabType} tab)`);
            
            const formData = new FormData();
            formData.append('csvFile', file);
            
            const response = await fetch(`${this.apiBase}/csv/upload-ad-copy`, {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.message || 'CSV upload failed');
            }
            
            console.log('✅ CSV processed successfully:', result);
            this.sheetsAdCopy = result.adCopy;
            
            // Show preview based on tab type
            if (tabType === 'duplicate') {
                this.displayCSVPreviewDuplicate(result.adCopy);
                this.updateAdSetCreationInfo();
            } else {
                this.displayCSVPreview(result.adCopy);
                this.updateCreateAdsButton();
            }
            
            this.showAlert(`Successfully loaded ${result.totalVariations} ad copy variations`, 'success');
            
        } catch (error) {
            console.error('❌ Error uploading CSV:', error);
            this.showAlert('Error processing CSV: ' + error.message, 'error');
        }
    }

    displayCSVPreview(data) {
        const previewElement = document.getElementById('csv-preview');
        const previewDataElement = document.getElementById('csv-data-preview');
        
        if (data.length === 0) {
            previewElement.classList.add('hidden');
            return;
        }

        // Show first 3 rows as preview
        const previewData = data.slice(0, 3);
        const previewHtml = `
            <div style="font-size: 0.875rem;">
                <strong>${data.length} ad copy variations loaded</strong>
                ${previewData.map((item, index) => `
                    <div style="margin: 0.5rem 0; padding: 0.5rem; background: white; border-radius: 4px; border: 1px solid #e2e8f0;">
                        <div><strong>Book:</strong> ${item.bookId} | <strong>Variation:</strong> ${item.variation}</div>
                        <div><strong>Primary:</strong> ${item.primaryText.substring(0, 80)}${item.primaryText.length > 80 ? '...' : ''}</div>
                        <div><strong>Headline:</strong> ${item.headline}</div>
                    </div>
                `).join('')}
                ${data.length > 3 ? `<div style="text-align: center; color: #718096; font-style: italic;">...and ${data.length - 3} more</div>` : ''}
            </div>
        `;
        
        previewDataElement.innerHTML = previewHtml;
        previewElement.classList.remove('hidden');
    }

    updateCreateAdsButton() {
        const button = document.getElementById('create-ads');
        const totalCombinations = this.uploadedFiles.length * this.sheetsAdCopy.length;
        
        if (totalCombinations > 0) {
            button.disabled = false;
            button.textContent = `Create ${totalCombinations} Ad Combinations`;
        } else {
            button.disabled = true;
            button.textContent = 'Create All Ad Combinations';
        }
    }

    // File upload handling
    handleFileUpload(files) {
        console.log('📤 handleFileUpload() called with files:', files.length);

        // Make upload cumulative - add new files to existing ones
        const newFiles = Array.from(files);

        // Filter out duplicates based on file name and size
        const existingFileKeys = new Set(
            this.uploadedFiles.map(f => `${f.name}_${f.size}`)
        );

        const uniqueNewFiles = newFiles.filter(file => {
            const fileKey = `${file.name}_${file.size}`;
            return !existingFileKeys.has(fileKey);
        });

        if (uniqueNewFiles.length < newFiles.length) {
            const duplicateCount = newFiles.length - uniqueNewFiles.length;
            console.log(`⚠️ Skipped ${duplicateCount} duplicate file(s)`);
            this.showAlert(`Skipped ${duplicateCount} duplicate file(s)`, 'warning');
        }

        // Add unique new files to existing files
        this.uploadedFiles = [...this.uploadedFiles, ...uniqueNewFiles];

        // Update both tabs' displays
        this.updateFileDisplay();
        this.updateCreateAdsButton();
        this.updateAdSetCreationInfo();

        console.log('✅ Total files uploaded:', this.uploadedFiles.map(f => f.name));
    }

    updateFileDisplay() {
        // Update original tab
        const uploadedFilesElement = document.getElementById('uploaded-files');
        const filesListElement = document.getElementById('files-list');
        
        // Update duplicate tab
        const uploadedFilesElementDuplicate = document.getElementById('uploaded-files-duplicate');
        const filesListElementDuplicate = document.getElementById('files-list-duplicate');
        
        if (this.uploadedFiles.length === 0) {
            if (uploadedFilesElement) uploadedFilesElement.classList.add('hidden');
            if (uploadedFilesElementDuplicate) uploadedFilesElementDuplicate.classList.add('hidden');
            return;
        }

        const filesHtml = this.uploadedFiles.map((file, index) => `
            <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.5rem; background: #f7fafc; border: 1px solid #e2e8f0; border-radius: 4px; margin-bottom: 0.5rem;">
                <div>
                    <strong>${file.name}</strong>
                    <span style="color: #718096; margin-left: 0.5rem;">(${(file.size / 1024 / 1024).toFixed(2)} MB)</span>
                </div>
                <button class="remove-file-btn" data-index="${index}" style="background: #fed7d7; color: #c53030; border: none; padding: 0.25rem 0.5rem; border-radius: 4px; cursor: pointer;">Remove</button>
            </div>
        `).join('');

        // Update both tabs
        if (filesListElement) {
            filesListElement.innerHTML = filesHtml;
            uploadedFilesElement.classList.remove('hidden');
            // Add event listeners to remove buttons
            this.attachRemoveButtonListeners(filesListElement);
        }

        if (filesListElementDuplicate) {
            filesListElementDuplicate.innerHTML = filesHtml;
            uploadedFilesElementDuplicate.classList.remove('hidden');
            // Add event listeners to remove buttons
            this.attachRemoveButtonListeners(filesListElementDuplicate);
        }
    }

    removeFile(index) {
        console.log(`🗑️ Removing file at index ${index}`);
        this.uploadedFiles.splice(index, 1);
        this.updateFileDisplay();
        this.updateCreateAdsButton();
        this.updateAdSetCreationInfo();
        this.showAlert('File removed', 'success');
    }

    attachRemoveButtonListeners(container) {
        const removeButtons = container.querySelectorAll('.remove-file-btn');
        removeButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                e.stopPropagation();
                const index = parseInt(button.getAttribute('data-index'));
                this.removeFile(index);
            });
        });
    }

    showAlert(message, type = 'info') {
        // Create alert element
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type}`;
        alertDiv.textContent = message;
        
        // Insert at the top of the container
        const container = document.querySelector('.container');
        container.insertBefore(alertDiv, container.firstChild);
        
        // Auto-remove after 5 seconds
        setTimeout(() => {
            if (alertDiv.parentNode) {
                alertDiv.parentNode.removeChild(alertDiv);
            }
        }, 5000);
    }


    downloadSampleCSV() {
        // Use the API route to download the sample CSV
        const link = document.createElement('a');
        link.href = `${this.apiBase}/csv/sample-format`;
        link.download = 'ad-copy-sample.csv';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

    // Duplicate CSV handling functions (using same API route)
    loadCSVDataDuplicate() {
        console.log('📄 loadCSVDataDuplicate() called');
        const fileInput = document.getElementById('csv-file-input-duplicate');
        if (fileInput && fileInput.files.length > 0) {
            this.uploadCSVFile(fileInput.files[0], 'duplicate');
        } else {
            this.showAlert('Please select a CSV file first', 'error');
        }
    }

    handleCSVFileSelectDuplicate(file) {
        console.log('📄 handleCSVFileSelectDuplicate() called with file:', file.name);
        
        const csvUploadArea = document.getElementById('csv-upload-area-duplicate');
        const fileInput = csvUploadArea.querySelector('#csv-file-input-duplicate');
        
        // Update upload area display - preserve the file input element
        const paragraphs = csvUploadArea.querySelectorAll('p');
        paragraphs.forEach(p => p.remove());
        
        const newParagraph = document.createElement('p');
        newParagraph.innerHTML = `📄 Selected: <strong>${file.name}</strong> (${(file.size / 1024).toFixed(1)} KB)`;
        csvUploadArea.insertBefore(newParagraph, fileInput);
        
        // Store the file for later upload when user clicks the button
        console.log('📄 File stored (duplicate), ready for upload when user clicks Upload CSV button');
    }

    displayCSVPreviewDuplicate(data) {
        const previewElement = document.getElementById('csv-preview-duplicate');
        const previewDataElement = document.getElementById('csv-data-preview-duplicate');
        
        if (data.length === 0) {
            previewElement.classList.add('hidden');
            return;
        }

        const previewData = data.slice(0, 5);
        const rows = previewData.map(item => {
            const truncate = (s, n) => s ? (s.length > n ? s.substring(0, n) + '...' : s) : '—';
            return `
                <div style="margin: 0.5rem 0; padding: 0.6rem; background: white; border-radius: 4px; border: 1px solid #e2e8f0; font-size:0.8rem;">
                    <div style="display:flex; gap:1rem; flex-wrap:wrap; margin-bottom:0.3rem;">
                        ${item.adsetName ? `<span><strong>AdSet:</strong> ${item.adsetName}</span>` : ''}
                        ${item.startDateTime ? `<span><strong>Start:</strong> ${item.startDateTime}</span>` : ''}
                        ${item.adName ? `<span><strong>Ad:</strong> ${item.adName}</span>` : ''}
                    </div>
                    <div><strong>Primary:</strong> ${truncate(item.primaryText, 80)}</div>
                    <div><strong>Headline:</strong> ${truncate(item.headline, 50)} &nbsp;|&nbsp; <strong>Desc:</strong> ${truncate(item.description, 40)}</div>
                    <div><strong>URL:</strong> ${truncate(item.landingPageUrl, 60)}</div>
                    <div style="display:flex; gap:1rem; flex-wrap:wrap;">
                        <span><strong>Video:</strong> ${item.videoUrl ? '✅ ' + truncate(item.videoUrl, 40) : '—'}</span>
                        <span><strong>Thumb:</strong> ${item.thumbnailUrl ? '✅' : '—'}</span>
                    </div>
                </div>`;
        }).join('');

        previewDataElement.innerHTML = `
            <div style="font-size: 0.875rem;">
                <strong>${data.length} row(s) loaded</strong>
                ${rows}
                ${data.length > 5 ? `<div style="text-align:center;color:#718096;font-style:italic;">...and ${data.length - 5} more</div>` : ''}
            </div>`;

        previewElement.classList.remove('hidden');
    }


    // Tab navigation functions
    setupTabNavigation() {
        // Main tabs
        document.querySelectorAll('.tab[data-tab]').forEach(tab => {
            tab.addEventListener('click', (e) => {
                const tabId = e.target.getAttribute('data-tab');
                this.showTab(tabId);
            });
        });
    }

    showTab(tabId) {
        // Main tabs
        if (tabId === 'creative-ads' || tabId === 'performance-monitoring') {
            // Hide all main tab contents
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            
            // Remove active from all main tabs
            document.querySelectorAll('.tab[data-tab]').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Show selected main tab
            document.getElementById(`${tabId}-tab`).classList.add('active');
            document.querySelector(`[data-tab="${tabId}"]`).classList.add('active');
        }
        
        // Sub-tabs within creative-ads
        if (tabId === 'add-to-existing' || tabId === 'create-new-adset') {
            // Hide all sub-tab contents
            document.getElementById('add-to-existing-tab').classList.remove('active');
            document.getElementById('create-new-adset-tab').classList.remove('active');
            
            // Remove active from all sub-tabs
            document.querySelectorAll('.tab[data-tab="add-to-existing"], .tab[data-tab="create-new-adset"]').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Show selected sub-tab
            document.getElementById(`${tabId}-tab`).classList.add('active');
            document.querySelector(`[data-tab="${tabId}"]`).classList.add('active');
        }
        
        // Sub-tabs within performance-monitoring - removed
    }

    // Duplicate adset functions
    async loadAdSetsDuplicate(campaignId) {
        if (!campaignId) {
            document.getElementById('adset-select-duplicate').innerHTML = '<option value="">Select a campaign first</option>';
            document.getElementById('adset-select-duplicate').disabled = true;
            document.getElementById('reference-ad-select-duplicate').innerHTML = '<option value="">Select an adset first</option>';
            document.getElementById('reference-ad-select-duplicate').disabled = true;
            return;
        }

        try {
            const response = await fetch(`${this.apiBase}/campaigns/${campaignId}`);
            const data = await response.json();
            
            const adsetSelect = document.getElementById('adset-select-duplicate');
            adsetSelect.innerHTML = '<option value="">Select an adset to duplicate...</option>';
            
            data.adsets.forEach(adset => {
                const option = document.createElement('option');
                option.value = adset.id;
                option.textContent = `${adset.name} (${adset.effective_status})`;
                adsetSelect.appendChild(option);
            });
            
            adsetSelect.disabled = false;
            
        } catch (error) {
            console.error('Error loading adsets for duplicate:', error);
            this.showAlert('Error loading adsets: ' + error.message, 'error');
        }
    }

    async loadReferenceAdsDuplicate(adsetId) {
        if (!adsetId) {
            document.getElementById('reference-ad-select-duplicate').innerHTML = '<option value="">Select an adset first</option>';
            document.getElementById('reference-ad-select-duplicate').disabled = true;
            return;
        }

        try {
            const response = await fetch(`${this.apiBase}/campaigns/adset/${adsetId}/ads`);
            const data = await response.json();
            
            const referenceAdSelect = document.getElementById('reference-ad-select-duplicate');
            referenceAdSelect.innerHTML = '<option value="">Select a reference ad...</option>';
            
            data.ads.forEach(ad => {
                const option = document.createElement('option');
                option.value = ad.id;
                option.textContent = `${ad.name} (${ad.effective_status})`;
                referenceAdSelect.appendChild(option);
            });
            
            referenceAdSelect.disabled = false;
            
        } catch (error) {
            console.error('Error loading reference ads for duplicate:', error);
            this.showAlert('Error loading reference ads: ' + error.message, 'error');
        }
    }

    updateAdSetCreationInfo() {
        const hasVideoUrls = this.sheetsAdCopy.some(row => row.videoUrl);
        const totalAds = hasVideoUrls
            ? this.sheetsAdCopy.length
            : this.uploadedFiles.length * this.sheetsAdCopy.length;

        // Count unique adset names (each unique name = one adset)
        const uniqueAdsetNames = new Set(this.sheetsAdCopy.map(row => row.adsetName || '__default__'));
        const adsetsNeeded = hasVideoUrls
            ? uniqueAdsetNames.size
            : Math.ceil(totalAds / 50);

        document.getElementById('total-combinations').textContent = adsetsNeeded;
        document.getElementById('adsets-needed').textContent = totalAds;

        const button = document.getElementById('create-duplicate-adset');
        if (totalAds > 0) {
            button.disabled = false;
            button.textContent = `Create ${adsetsNeeded} AdSet${adsetsNeeded > 1 ? 's' : ''} & ${totalAds} Ad${totalAds > 1 ? 's' : ''}`;
        } else {
            button.disabled = true;
            button.textContent = 'Create New AdSet & Ads';
        }
    }

    async createDuplicateAdSet() {
        console.log('🚀 createDuplicateAdSet() called');
        
        try {
            // Get selected values
            const campaignId = document.getElementById('campaign-select-duplicate').value;
            const referenceAdsetId = document.getElementById('adset-select-duplicate').value;
            const referenceAdId = document.getElementById('reference-ad-select-duplicate').value;

            console.log('📋 Duplicate adset values:', { campaignId, referenceAdsetId, referenceAdId, csvRows: this.sheetsAdCopy.length });
            
            if (!campaignId) {
                alert('Please select a Campaign first');
                return;
            }
            if (!referenceAdsetId) {
                alert('Please select a Reference AdSet first');
                return;
            }
            if (!referenceAdId) {
                alert('Please select a Reference Ad first');
                return;
            }
            
            if (this.sheetsAdCopy.length === 0) {
                alert('Please upload CSV ad copy data first');
                return;
            }

            const button = document.getElementById('create-duplicate-adset');
            const originalText = button.innerHTML;
            button.innerHTML = '⏳ Creating duplicate adset(s)...';
            button.disabled = true;
            
            const hasVideoUrls = this.sheetsAdCopy.some(row => row.videoUrl);
            const totalCombinations = hasVideoUrls
                ? this.sheetsAdCopy.length
                : this.uploadedFiles.length * this.sheetsAdCopy.length;
            const adsetsNeeded = Math.ceil(totalCombinations / 50);
            
            console.log(`📊 Creating ${adsetsNeeded} adset(s) for ${totalCombinations} ad combinations`);
            
            // Step 1: Upload creative files if no videoUrl in CSV
            let uploadResult = { success: true, creativeIds: [], files: [] };

            if (!hasVideoUrls) {
                if (this.uploadedFiles.length === 0) {
                    this.showAlert('Please upload creative files or add VideoURL to your CSV', 'error');
                    button.innerHTML = originalText;
                    button.disabled = false;
                    return;
                }
                console.log('📤 Uploading creative files to server...');
                const formData = new FormData();
                this.uploadedFiles.forEach(file => {
                    formData.append('creatives', file);
                });
                formData.append('adsetId', referenceAdsetId);
                formData.append('adAccountId', this.selectedAdAccountId);

                const uploadResponse = await fetch(`${this.apiBase}/creatives/upload-for-adset`, {
                    method: 'POST',
                    body: formData
                });
                uploadResult = await uploadResponse.json();

                if (!uploadResult.success || uploadResult.creativeIds.length === 0) {
                    throw new Error('Failed to upload creative files: ' + (uploadResult.error || 'No creative IDs returned'));
                }
                console.log('✅ Creative files uploaded successfully. Meta hashes:', uploadResult.creativeIds);
            } else {
                console.log('📋 CSV has videoUrls - skipping file upload, videos will be uploaded per row');
            }

            // Create filename mapping: creativeId -> originalName
            const creativeFilenames = {};
            if (uploadResult.files) {
                uploadResult.files.forEach(file => {
                    const creativeId = file.metaHash || file.metaVideoId;
                    if (creativeId && file.originalName) {
                        creativeFilenames[creativeId] = file.originalName;
                    }
                });
            }
            console.log('📁 Creative filename mapping:', creativeFilenames);

            // Step 2: Create duplicate adset(s) with ads
            console.log('🎯 Creating duplicate adset(s) and ads...');
            const createResponse = await fetch(`${this.apiBase}/campaigns/create-duplicate-adset`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    campaignId: campaignId,
                    referenceAdsetId: referenceAdsetId,
                    referenceAdId: referenceAdId,
                    creativeIds: uploadResult.creativeIds, // Meta image hashes
                    adCopyVariations: this.sheetsAdCopy,
                    maxAdsPerAdset: 50,
                    creativeFilenames,
                    adAccountId: this.selectedAdAccountId  // Send selected ad account
                })
            });
            
            const createResult = await createResponse.json();
            
            if (!createResult.success) {
                throw new Error('Failed to create duplicate adset: ' + (createResult.error || 'Unknown error'));
            }
            
            // Success! Show results
            const totalAds = createResult.totalAdsCreated || 0;
            const adsetsCreated = createResult.adsetsCreated || 0;
            const failedAds = createResult.failedAds || 0;
            
            let message = `✅ Successfully created ${adsetsCreated} duplicate adset(s) with ${totalAds} ads`;
            if (failedAds > 0) {
                message += ` (${failedAds} ads failed)`;
            }
            
            this.showAlert(message, 'success');
            
            console.log('🎉 Duplicate adset creation completed:', {
                adsetsCreated: adsetsCreated,
                totalAdsCreated: totalAds,
                failedAds: failedAds,
                results: createResult.results
            });
            
        } catch (error) {
            console.error('❌ Error creating duplicate adset:', error);
            this.showAlert('Error creating duplicate adset: ' + error.message, 'error');
        } finally {
            const button = document.getElementById('create-duplicate-adset');
            button.innerHTML = 'Create New AdSet & Ads';
            button.disabled = false;
        }
    }

}


// Initialize the dashboard
document.addEventListener('DOMContentLoaded', () => {
    window.dashboard = new MetaAdsDashboard();
});
