const express = require('express');
const multer = require('multer');
const path = require('path');
const fs = require('fs').promises;
const sharp = require('sharp');
const { v4: uuidv4 } = require('uuid');
const { FacebookAdsApi, AdAccount, AdImage } = require('facebook-nodejs-business-sdk');
const router = express.Router();

// Initialize Facebook API
FacebookAdsApi.init(process.env.META_ACCESS_TOKEN);

// Meta Ad Account for uploading creatives
const account = new AdAccount(process.env.META_AD_ACCOUNT_ID);

console.log(`📋 Meta API initialized with account: ${process.env.META_AD_ACCOUNT_ID}`);

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: async (req, file, cb) => {
    // For upload-for-adset endpoint, use a simpler path
    const uploadPath = path.join(__dirname, '..', 'uploads', 'temp');

    try {
      await fs.mkdir(uploadPath, { recursive: true });
      cb(null, uploadPath);
    } catch (error) {
      console.error('Error creating upload directory:', error);
      cb(error);
    }
  },
  filename: (req, file, cb) => {
    const uniqueId = uuidv4();
    const extension = path.extname(file.originalname);
    const timestamp = Date.now();
    cb(null, `${timestamp}_${uniqueId}${extension}`);
  }
});

const fileFilter = (req, file, cb) => {
  console.log(`🔍 Multer processing field: "${file.fieldname}" for file: ${file.originalname}`);

  const allowedTypes = process.env.ALLOWED_FILE_TYPES?.split(',') || ['jpg', 'jpeg', 'png', 'mp4', 'mov'];
  const fileExtension = path.extname(file.originalname).slice(1).toLowerCase();

  if (allowedTypes.includes(fileExtension)) {
    cb(null, true);
  } else {
    cb(new Error(`File type ${fileExtension} not allowed. Allowed types: ${allowedTypes.join(', ')}`), false);
  }
};

const upload = multer({
  storage: storage,
  fileFilter: fileFilter,
  limits: {
    fileSize: (process.env.MAX_FILE_SIZE_MB || 250) * 1024 * 1024 // Convert MB to bytes (250MB default)
  }
});

// Upload creative files
router.post('/upload', upload.array('creatives', 20), async (req, res) => {
  try {
    const { bookId, campaignType, adCopySheet } = req.body;
    const uploadedFiles = [];

    if (!req.files || req.files.length === 0) {
      return res.status(400).json({ error: 'No files uploaded' });
    }

    for (const file of req.files) {
      const fileInfo = {
        id: uuidv4(),
        originalName: file.originalname,
        filename: file.filename,
        path: file.path,
        size: file.size,
        mimetype: file.mimetype,
        bookId: bookId,
        campaignType: campaignType,
        uploadedAt: new Date().toISOString(),
        status: 'pending',
        dimensions: null,
        duration: null
      };

      // Get image/video metadata
      if (file.mimetype.startsWith('image/')) {
        try {
          const metadata = await sharp(file.path).metadata();
          fileInfo.dimensions = {
            width: metadata.width,
            height: metadata.height,
            aspectRatio: (metadata.width / metadata.height).toFixed(2)
          };
        } catch (error) {
          console.error('Error getting image metadata:', error);
        }
      }

      uploadedFiles.push(fileInfo);
    }

    // Save upload info to database/file system
    await saveUploadInfo(uploadedFiles, { bookId, campaignType, adCopySheet });

    res.json({
      message: 'Files uploaded successfully',
      files: uploadedFiles,
      totalFiles: uploadedFiles.length
    });

  } catch (error) {
    console.error('Upload error:', error);
    res.status(500).json({ error: error.message });
  }
});

// Upload creative files for specific adset (for ad creation workflow)
router.post('/upload-for-adset', (req, res, next) => {
  console.log('🔥 DEBUG: /upload-for-adset endpoint hit!');
  console.log('📋 Request headers:', req.headers);

  upload.array('creatives', 100)(req, res, (err) => {  // Increased from 20 to 100 files
    if (err) {
      console.error('❌ Multer error:', err);
      console.error('   Error code:', err.code);
      console.error('   Error field:', err.field);
      console.error('   Error message:', err.message);

      if (err.code === 'LIMIT_UNEXPECTED_FILE') {
        return res.status(400).json({
          error: `Unexpected field name: "${err.field}". Expected field name is "creatives"`,
          details: 'Please ensure file uploads use field name "creatives"'
        });
      }

      return res.status(400).json({ error: err.message });
    }
    next();
  });
}, async (req, res) => {
  try {
    const { adsetId, adAccountId } = req.body;
    const uploadedFiles = [];

    if (!req.files || req.files.length === 0) {
      return res.status(400).json({ error: 'No files uploaded' });
    }

    if (!adsetId) {
      return res.status(400).json({ error: 'AdSet ID is required' });
    }

    // Use selected ad account or default from .env, ensure act_ prefix
    const rawAccountId = (!adAccountId || adAccountId === 'null') ? process.env.META_AD_ACCOUNT_ID : adAccountId;
    const activeAdAccountId = rawAccountId && !rawAccountId.startsWith('act_') ? `act_${rawAccountId}` : rawAccountId;
    console.log(`🔑 ENV: ${process.env.META_AD_ACCOUNT_ID}, from request: ${adAccountId}, resolved: ${activeAdAccountId}`);
    const activeAccount = new AdAccount(activeAdAccountId);
    console.log(`📁 Uploading ${req.files.length} files for adset ${adsetId} to ad account ${activeAdAccountId}`);

    for (const file of req.files) {
      try {
        let metaHash = null;
        let metaVideoId = null;
        let metaThumbnailUrl = null;

        // Upload image to Meta if it's an image file
        if (file.mimetype.startsWith('image/')) {
          console.log(`🖼️ Uploading image to Meta: ${file.originalname}`);
          
          try {
            // Debug: Log what we're trying to upload
            console.log(`📋 File details:`, {
              path: file.path,
              originalname: file.originalname,
              mimetype: file.mimetype,
              size: file.size
            });
            
            // Try real Meta image upload with proper error handling
            try {
              const fs = require('fs');
              
              // Method 1: Try with file stream
              console.log(`🔄 Attempting Meta upload with file stream...`);
              const imageData = await activeAccount.createAdImage([], {
                filename: fs.createReadStream(file.path)
              });
              
              console.log(`🔍 Raw imageData response (stream):`, JSON.stringify(imageData, null, 2));
              
              // Try different ways to extract the hash based on the API response structure
              metaHash = imageData.hash || 
                        imageData.image_hash || 
                        imageData.id || 
                        imageData._data?.hash ||
                        imageData.images?.bytes?.hash ||
                        imageData._data?.images?.bytes?.hash ||
                        imageData._changes?.images?.bytes?.hash;
              
              if (!metaHash) {
                throw new Error('Meta API returned success but no hash found in response');
              }
              
              console.log(`✅ Meta image upload successful, hash: ${metaHash}`);
              
            } catch (streamError) {
              console.log(`⚠️ Stream upload failed: ${streamError.message}`);
              
              try {
                // Method 2: Try with base64 encoded data
                console.log(`🔄 Attempting Meta upload with base64 data...`);
                const fileBuffer = await fs.readFile(file.path);
                const base64Data = fileBuffer.toString('base64');

                const imageData = await activeAccount.createAdImage([], {
                  bytes: base64Data
                });
                
                console.log(`🔍 Raw imageData response:`, JSON.stringify(imageData, null, 2));
                
                // Try different ways to extract the hash based on the API response structure
                metaHash = imageData.hash || 
                          imageData.image_hash || 
                          imageData.id || 
                          imageData._data?.hash ||
                          imageData.images?.bytes?.hash ||
                          imageData._data?.images?.bytes?.hash ||
                          imageData._changes?.images?.bytes?.hash;
                
                if (!metaHash) {
                  throw new Error('Meta API returned success but no hash found in response');
                }
                
                console.log(`✅ Meta image upload with base64 successful, hash: ${metaHash}`);
                
              } catch (base64Error) {
                console.log(`⚠️ Base64 upload failed: ${base64Error.message}`);
                
                try {
                  // Method 3: Try with raw buffer
                  console.log(`🔄 Attempting Meta upload with raw buffer...`);
                  const fileBuffer = await fs.readFile(file.path);

                  const imageData = await activeAccount.createAdImage([], {
                    bytes: fileBuffer
                  });
                  
                  console.log(`🔍 Raw imageData response (buffer):`, JSON.stringify(imageData, null, 2));
                  
                  // Try different ways to extract the hash based on the API response structure
                  metaHash = imageData.hash || 
                            imageData.image_hash || 
                            imageData.id || 
                            imageData._data?.hash ||
                            imageData.images?.bytes?.hash ||
                            imageData._data?.images?.bytes?.hash ||
                            imageData._changes?.images?.bytes?.hash;
                  
                  if (!metaHash) {
                    throw new Error('Meta API returned success but no hash found in response');
                  }
                  
                  console.log(`✅ Meta image upload with buffer successful, hash: ${metaHash}`);
                  
                } catch (bufferError) {
                  console.log(`❌ All upload methods failed. Last error: ${bufferError.message}`);
                  
                  // Final fallback: Use existing image hash from reference ad
                  try {
                    const { adsetId } = req.body;
                    const metaService = require('../services/metaService');
                    const adsetAds = await metaService.getAdSetAds(adsetId);
                    
                    if (adsetAds && adsetAds.length > 0) {
                      console.log(`🔍 Found ${adsetAds.length} existing ads, extracting image hash...`);
                      
                      for (const ad of adsetAds) {
                        const adDetails = await metaService.getAdDetails(ad.id);
                        if (adDetails?.creative?.object_story_spec?.link_data?.image_hash) {
                          metaHash = adDetails.creative.object_story_spec.link_data.image_hash;
                          console.log(`✅ Using existing image hash from ad ${ad.id}: ${metaHash}`);
                          break;
                        }
                      }
                    }
                    
                    if (!metaHash) {
                      throw new Error('Unable to upload image to Meta and no existing image hash found');
                    }
                  } catch (fallbackError) {
                    throw new Error(`Meta image upload failed: ${bufferError.message}`);
                  }
                }
              }
            }
            
          } catch (metaError) {
            console.error(`❌ Failed to upload image to Meta: ${metaError.message}`);
            throw new Error(`Meta image upload failed: ${metaError.message}`);
          }
        }

        // Upload video to Meta if it's a video file
        else if (file.mimetype.startsWith('video/')) {
          console.log(`🎥 Uploading video to Meta: ${file.originalname}`);

          try {
            const videoUploadService = require('../services/videoUploadService');
            const fileSizeInMB = (file.size / (1024 * 1024)).toFixed(2);
            console.log(`📊 Video size: ${fileSizeInMB} MB`);

            // Upload video and get video ID + thumbnail URL
            const videoData = await videoUploadService.uploadVideoToMeta(file.path, file.originalname, activeAdAccountId);
            metaVideoId = videoData.videoId;
            metaThumbnailUrl = videoData.thumbnailUrl;

            console.log(`✅ Video uploaded successfully! Video ID: ${metaVideoId}`);
            if (metaThumbnailUrl) {
              console.log(`✅ Auto-generated thumbnail: ${metaThumbnailUrl}`);
            }

          } catch (videoError) {
            console.error(`❌ Failed to upload video to Meta: ${videoError.message}`);
            throw new Error(`Meta video upload failed: ${videoError.message}`);
          }
        }

        const fileInfo = {
          id: uuidv4(),
          originalName: file.originalname,
          filename: file.filename,
          path: file.path,
          size: file.size,
          mimetype: file.mimetype,
          adsetId: adsetId,
          metaHash: metaHash, // For images
          metaVideoId: metaVideoId, // For videos
          metaThumbnailUrl: metaThumbnailUrl, // For video thumbnails
          uploadedAt: new Date().toISOString(),
          status: 'ready'
        };

        // Get image/video metadata
        if (file.mimetype.startsWith('image/')) {
          try {
            const metadata = await sharp(file.path).metadata();
            fileInfo.dimensions = {
              width: metadata.width,
              height: metadata.height,
              aspectRatio: (metadata.width / metadata.height).toFixed(2)
            };
          } catch (error) {
            console.error('Error getting image metadata:', error);
          }
        }

        console.log(`📝 Mapping: creativeId="${fileInfo.metaVideoId || fileInfo.metaHash}" -> originalName="${fileInfo.originalName}"`);
        uploadedFiles.push(fileInfo);
        
      } catch (error) {
        console.error(`❌ Error processing file ${file.originalname}:`, error);
        // Continue with other files even if one fails
        uploadedFiles.push({
          id: uuidv4(),
          originalName: file.originalname,
          error: error.message,
          status: 'failed'
        });
      }
    }

    console.log(`✅ Successfully processed ${uploadedFiles.length} files for adset ${adsetId}`);

    // Filter successful uploads
    const successfulUploads = uploadedFiles.filter(f => f.status === 'ready' && (f.metaHash || f.metaVideoId));
    const failedUploads = uploadedFiles.filter(f => f.status === 'failed');

    res.json({
      success: successfulUploads.length > 0,
      message: `Creative files processed: ${successfulUploads.length} successful, ${failedUploads.length} failed`,
      creativeIds: successfulUploads.map(f => f.metaHash || f.metaVideoId), // Return Meta hashes/video IDs for ad creation
      files: uploadedFiles,
      totalFiles: uploadedFiles.length,
      successfulFiles: successfulUploads.length,
      failedFiles: failedUploads.length
    });

  } catch (error) {
    console.error('Upload for adset error:', error);
    res.status(500).json({ 
      success: false,
      error: error.message 
    });
  }
});

// Get creative folders and files
router.get('/folders', async (req, res) => {
  try {
    const uploadsDir = path.join(__dirname, '..', 'uploads');
    const folders = await getCreativeFolders(uploadsDir);
    res.json({ folders });
  } catch (error) {
    console.error('Error getting folders:', error);
    res.status(500).json({ error: error.message });
  }
});

// Get files in a specific folder
router.get('/folder/:bookId/:campaignType?', async (req, res) => {
  try {
    const { bookId, campaignType } = req.params;
    const folderPath = path.join(__dirname, '..', 'uploads', bookId, campaignType || '');
    const files = await getFilesInFolder(folderPath);
    res.json({ files });
  } catch (error) {
    console.error('Error getting folder files:', error);
    res.status(500).json({ error: error.message });
  }
});

// Delete a creative file
router.delete('/:fileId', async (req, res) => {
  try {
    const { fileId } = req.params;
    await deleteCreativeFile(fileId);
    res.json({ message: 'File deleted successfully' });
  } catch (error) {
    console.error('Error deleting file:', error);
    res.status(500).json({ error: error.message });
  }
});

// Test Meta image upload endpoint
router.post('/test-meta-upload', upload.single('test_image'), async (req, res) => {
  try {
    console.log('🧪 Testing Meta image upload...');
    
    if (!req.file) {
      return res.status(400).json({ error: 'No file provided for testing' });
    }

    console.log('📋 File details:', {
      originalname: req.file.originalname,
      mimetype: req.file.mimetype,
      size: req.file.size,
      path: req.file.path
    });

    let metaHash = null;
    const fs = require('fs');

    try {
      // Test with file stream
      console.log('🔄 Testing stream upload...');
      const imageData = await account.createAdImage([], {
        filename: fs.createReadStream(req.file.path)
      });
      
      console.log('🔍 Test response (stream):', JSON.stringify(imageData, null, 2));
      metaHash = imageData.hash || 
                imageData.image_hash || 
                imageData.id || 
                imageData._data?.hash ||
                imageData.images?.bytes?.hash ||
                imageData._data?.images?.bytes?.hash ||
                imageData._changes?.images?.bytes?.hash;
      console.log('✅ Stream upload successful:', metaHash);
      
      return res.json({
        success: true,
        method: 'stream',
        hash: metaHash,
        message: 'Meta image upload test successful'
      });
      
    } catch (streamError) {
      console.log('⚠️ Stream upload failed:', streamError.message);
      
      try {
        // Test with buffer
        console.log('🔄 Testing buffer upload...');
        const fileBuffer = await fs.readFile(req.file.path);
        
        const imageData = await account.createAdImage([], {
          bytes: fileBuffer
        });
        
        console.log('🔍 Test response (buffer):', JSON.stringify(imageData, null, 2));
        metaHash = imageData.hash || 
                  imageData.image_hash || 
                  imageData.id || 
                  imageData._data?.hash ||
                  imageData.images?.bytes?.hash ||
                  imageData._data?.images?.bytes?.hash ||
                  imageData._changes?.images?.bytes?.hash;
        console.log('✅ Buffer upload successful:', metaHash);
        
        return res.json({
          success: true,
          method: 'buffer',
          hash: metaHash,
          message: 'Meta image upload test successful (buffer method)'
        });
        
      } catch (bufferError) {
        console.log('❌ Buffer upload failed:', bufferError.message);
        
        return res.json({
          success: false,
          errors: {
            stream: streamError.message,
            buffer: bufferError.message
          },
          message: 'Both Meta upload methods failed'
        });
      }
    }
    
  } catch (error) {
    console.error('❌ Test upload error:', error);
    res.status(500).json({
      success: false,
      error: error.message,
      message: 'Meta image upload test failed'
    });
  }
});

// Helper functions
async function saveUploadInfo(files, metadata) {
  const uploadLogPath = path.join(__dirname, '..', 'uploads', 'upload_log.json');
  
  try {
    let existingData = [];
    try {
      const data = await fs.readFile(uploadLogPath, 'utf8');
      existingData = JSON.parse(data);
    } catch (error) {
      // File doesn't exist yet, start with empty array
    }

    const uploadRecord = {
      id: uuidv4(),
      timestamp: new Date().toISOString(),
      metadata: metadata,
      files: files
    };

    existingData.push(uploadRecord);
    await fs.writeFile(uploadLogPath, JSON.stringify(existingData, null, 2));
  } catch (error) {
    console.error('Error saving upload info:', error);
  }
}

async function getCreativeFolders(uploadsDir) {
  try {
    const items = await fs.readdir(uploadsDir, { withFileTypes: true });
    const folders = [];

    for (const item of items) {
      if (item.isDirectory() && item.name !== 'upload_log.json') {
        const bookPath = path.join(uploadsDir, item.name);
        const subFolders = await fs.readdir(bookPath, { withFileTypes: true });
        
        const campaignTypes = [];
        for (const subItem of subFolders) {
          if (subItem.isDirectory()) {
            const filesCount = await countFilesInFolder(path.join(bookPath, subItem.name));
            campaignTypes.push({
              name: subItem.name,
              filesCount: filesCount
            });
          }
        }

        folders.push({
          bookId: item.name,
          campaignTypes: campaignTypes,
          totalFiles: campaignTypes.reduce((sum, ct) => sum + ct.filesCount, 0)
        });
      }
    }

    return folders;
  } catch (error) {
    console.error('Error getting creative folders:', error);
    return [];
  }
}

async function getFilesInFolder(folderPath) {
  try {
    const files = await fs.readdir(folderPath);
    const fileDetails = [];

    for (const file of files) {
      const filePath = path.join(folderPath, file);
      const stats = await fs.stat(filePath);
      
      if (stats.isFile()) {
        fileDetails.push({
          name: file,
          path: filePath,
          size: stats.size,
          createdAt: stats.birthtime,
          modifiedAt: stats.mtime
        });
      }
    }

    return fileDetails;
  } catch (error) {
    console.error('Error getting files in folder:', error);
    return [];
  }
}

async function countFilesInFolder(folderPath) {
  try {
    const files = await fs.readdir(folderPath);
    return files.filter(file => !file.startsWith('.')).length;
  } catch (error) {
    return 0;
  }
}

async function deleteCreativeFile(fileId) {
  // Implementation for deleting files by ID
  // This would need to be integrated with your file tracking system
  throw new Error('Delete functionality not implemented yet');
}

module.exports = router;