const express = require('express');
const multer = require('multer');
const csv = require('csv-parser');
const fs = require('fs');
const path = require('path');
const router = express.Router();

const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const uploadDir = path.join(__dirname, '..', 'uploads', 'csv');
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, 'adcopy-' + uniqueSuffix + '.csv');
  }
});

const upload = multer({ 
  storage: storage,
  fileFilter: (req, file, cb) => {
    if (file.mimetype === 'text/csv' || file.originalname.endsWith('.csv')) {
      cb(null, true);
    } else {
      cb(new Error('Only CSV files are allowed'));
    }
  },
  limits: { fileSize: 5 * 1024 * 1024 }
});

router.post('/upload-ad-copy', upload.single('csvFile'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: 'No CSV file uploaded' });
    }

    const csvFilePath = req.file.path;
    const adCopyData = [];
    const allRows = [];

    const stream = fs.createReadStream(csvFilePath)
      .pipe(csv({
        headers: ['adsetName', 'startDateTime', 'adName', 'primaryText', 'headline', 'description', 'landingPageUrl', 'videoUrl', 'thumbnailUrl'],
        skipEmptyLines: true
      }));

    for await (const row of stream) {
      // Skip header row
      if (row.adsetName === 'AdsetName' || row.adsetName === 'adsetName') continue;
      // Skip empty rows
      if (!row.adName && !row.primaryText && !row.headline) continue;
      allRows.push(row);
    }

    for (const row of allRows) {
      adCopyData.push({
        adsetName: row.adsetName ? row.adsetName.trim() : '',
        startDateTime: row.startDateTime ? row.startDateTime.trim() : '',
        adName: row.adName ? row.adName.trim() : '',
        primaryText: row.primaryText || '',
        headline: row.headline || '',
        description: row.description || '',
        callToAction: '',
        landingPageUrl: row.landingPageUrl ? row.landingPageUrl.trim() : '',
        videoUrl: row.videoUrl ? row.videoUrl.trim() : '',
        thumbnailUrl: row.thumbnailUrl ? row.thumbnailUrl.trim() : ''
      });
    }

    fs.unlinkSync(csvFilePath);

    res.json({
      success: true,
      adCopy: adCopyData,
      totalVariations: adCopyData.length,
      message: `Successfully parsed ${adCopyData.length} ad copy variations from CSV`
    });

  } catch (error) {
    console.error('Error processing CSV file:', error);
    if (req.file && fs.existsSync(req.file.path)) fs.unlinkSync(req.file.path);
    res.status(500).json({ error: error.message, message: 'Error processing CSV file.' });
  }
});

router.get('/sample-format', (req, res) => {
  const sampleCSV = `AdsetName,StartDateTime,AdName,PrimaryText,Headline,Description,LandingPageURL,VideoURL,ThumbnailURL
My Adset 1,2026-03-01 08:00,My Ad 1,"Help kids smile again","Donate Today","Change a life","https://smilesreborn.org/","https://example.com/video1.mp4","https://example.com/thumb1.jpg"
My Adset 2,2026-03-01 08:00,My Ad 2,"Every child deserves to smile","Give Now","Make an impact","https://smilesreborn.org/page2","https://example.com/video2.mp4","https://example.com/thumb2.jpg"`;

  res.setHeader('Content-Type', 'text/csv');
  res.setHeader('Content-Disposition', 'attachment; filename="ad-copy-sample.csv"');
  res.send(sampleCSV);
});

module.exports = router;
