const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');
const path = require('path');
const crypto = require('crypto');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 3000;

// Auth config
const AUTH_USER = process.env.ADMIN_USERNAME || 'admin';
const AUTH_PASS = process.env.ADMIN_PASSWORD || 'changeme';
const TOKEN_SECRET = process.env.AUTH_SECRET || crypto.randomBytes(32).toString('hex');
const TOKEN_NAME = 'mads_auth';
const COOKIE_MAX_AGE = 60 * 60 * 24; // 24 hours in seconds

function signToken(val) {
  const sig = crypto.createHmac('sha256', TOKEN_SECRET).update(val).digest('hex');
  return `${val}.${sig}`;
}
function verifyToken(cookie) {
  if (!cookie) return false;
  const [val, sig] = cookie.split('.');
  const expected = crypto.createHmac('sha256', TOKEN_SECRET).update(val).digest('hex');
  return val === 'authenticated' && sig === expected;
}
function parseCookies(req) {
  const list = {};
  const rc = req.headers.cookie;
  if (rc) rc.split(';').forEach(c => {
    const parts = c.split('=');
    list[parts[0].trim()] = decodeURIComponent(parts.slice(1).join('=').trim());
  });
  return list;
}
function requireAuth(req, res, next) {
  const cookies = parseCookies(req);
  if (verifyToken(cookies[TOKEN_NAME])) return next();
  res.redirect('/login');
}

// Security middleware disabled (causes HTTPS redirect issues over HTTP)
// app.use(helmet());
app.use(cors());

// Rate limiting
const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100 // limit each IP to 100 requests per windowMs
});
app.use(limiter);

// Body parsing middleware
app.use(express.json({ limit: '250mb' }));
app.use(express.urlencoded({ extended: true, limit: '250mb' }));

// Static files
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));
app.use('/public', express.static(path.join(__dirname, 'public')));
app.use(express.static(path.join(__dirname, 'public')));

// Auth routes (unprotected)
app.get('/login', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'login.html'));
});
app.post('/auth/login', (req, res) => {
  const { username, password } = req.body;
  if (username === AUTH_USER && password === AUTH_PASS) {
    const token = signToken('authenticated');
    res.setHeader('Set-Cookie', `${TOKEN_NAME}=${encodeURIComponent(token)}; Path=/; HttpOnly; SameSite=Strict; Max-Age=${COOKIE_MAX_AGE}`);
    res.json({ ok: true });
  } else {
    res.status(401).json({ error: 'Invalid credentials' });
  }
});
app.get('/auth/logout', (req, res) => {
  res.setHeader('Set-Cookie', `${TOKEN_NAME}=; Path=/; HttpOnly; Max-Age=0`);
  res.redirect('/login');
});

// Protect all routes below
app.use(requireAuth);

// Proxy test - shows what IP the outside world sees
app.get('/api/proxy-test', async (req, res) => {
  const axiosInstance = require('./services/axiosInstance');
  const proxyUrl = process.env.PROXY_URL || '';
  let proxyHost = null;
  if (proxyUrl) { try { proxyHost = new URL(proxyUrl).hostname; } catch(e) { proxyHost = proxyUrl; } }

  const endpoints = ['https://api.ipify.org', 'https://ifconfig.me/ip', 'https://icanhazip.com'];
  for (const url of endpoints) {
    try {
      const response = await axiosInstance.get(url, { timeout: 10000 });
      const raw = response.data;
      const ip = typeof raw === 'string' ? raw.trim() : (raw?.ip || JSON.stringify(raw));
      console.log(`🌐 Proxy test via ${url} - IP: ${ip}`);
      return res.json({ ip, proxy_configured: !!proxyUrl, proxy_host: proxyHost, checked_via: url });
    } catch (e) {
      console.error(`Proxy test failed for ${url}: ${e.message}`);
    }
  }
  res.status(500).json({ error: 'All IP check endpoints failed', ip: null, proxy_configured: !!proxyUrl, proxy_host: proxyHost });
});

// Routes
app.use('/api/creatives', require('./routes/creatives'));
app.use('/api/campaigns', require('./routes/campaigns'));
app.use('/api/csv', require('./routes/csv'));
app.use('/api/monitoring', require('./routes/monitoring'));
app.use('/api/meta', require('./routes/meta'));
app.use('/api/duplication', require('./routes/duplication'));
app.use('/api/kpi', require('./routes/kpi'));

// Serve main dashboard
app.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error(err.stack);
  res.status(500).json({ 
    error: 'Something went wrong!', 
    message: process.env.NODE_ENV === 'development' ? err.message : 'Internal server error' 
  });
});

// 404 handler
app.use((req, res) => {
  res.status(404).json({ error: 'Route not found' });
});

// Start performance monitoring when server starts (disabled for debugging)
// const performanceMonitor = require('./services/performanceMonitor');
// performanceMonitor.start();

app.listen(PORT, () => {
  console.log(`🚀 Meta Ads Launcher running on port ${PORT}`);
  console.log(`📊 Dashboard: http://localhost:${PORT}`);
  console.log(`🔍 Performance monitoring active`);
});