/**
 * Copyright (c) 2017-present, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the license found in the
 * LICENSE file in the root directory of this source tree.
 * @flow
 */

import AttributionModel from './attribution-model';
import AttributionMethod from './attribution-method';
import DeclineReason from './decline-reason';
import AttributionSetting from './attribution-setting';

/**
 * AttributionData used for attribution passback event to optimize the performance.
 */
export default class AttributionData {
    _scope: string;
    _visit_time: number;
    _ad_id: string;
    _adset_id: string;
    _campaign_id: string;
    _attribution_share: number;
    _attribution_model: AttributionModel;
    _attr_window: number;
    _attribution_value: number;
    _attribution_source: string;
    _touchpoint_type: string;
    _touchpoint_ts: number;
    _attribution_method: AttributionMethod;
    _decline_reason: DeclineReason;
    _auditing_token: string;
    _linkage_key: string;
    _attribution_setting: AttributionSetting;


    /**
     * @param {String} scope Touchpoint type
     * @param {Number} visit_time Time that the campaign_id or fbc was first received
     * @param {String} ad_id Meta-provided ad id from URL/deeplink
     * @param {String} adset_id Meta-provided adset id from URL/deeplink
     * @param {String} campaign_id Meta-provided campaign id from URL/deeplink
     * @param {Number} attribution_share Range [0-1] weight of credit assigned to the visit
     * @param {AttributionModel} attribution_model Attribution model used to attribute the event, check attribution-model.js file.
     * @param {Number} attr_window Attribution window in days.
     * @param {Number} attribution_value The share of value generated by this click-conversion pair that is attributed to Meta.
     * @param {String} attribution_source The attribution source to differentiate the source of the data, e.g. whether this is from AMM or Custom Attribution or any other sources.
     * @param {String} touchpoint_type The engagement type that caused the original credited conversion.
     * @param {Number} touchpoint_ts The time when the touchpoint event occurred with the ad that the install was credited to.
     * @param {AttributionMethod} attribution_method The attribution method used to attribute the event, check attribution-method.js file.
     * @param {DeclineReason} decline_reason The decline reason for the attribution, check decline-reason.js file.
     * @param {String} auditing_token The auditing token for the attribution.
     * @param {String} linkage_key The linkage key for the attribution.
     * @param {AttributionSetting} attribution_setting The attribution setting configuration.
     */
    constructor(scope: string, visit_time: number, ad_id: string, adset_id: string, campaign_id: string,
        attribution_share: number, attribution_model: AttributionModel, attr_window: number, attribution_value: number,
        attribution_source: string, touchpoint_type: string, touchpoint_ts: number, attribution_method: AttributionMethod,
        decline_reason: DeclineReason, auditing_token: string, linkage_key: string, attribution_setting: AttributionSetting) {
        this._scope = scope;
        this._visit_time = visit_time;
        this._ad_id = ad_id;
        this._adset_id = adset_id;
        this._campaign_id = campaign_id;
        this._attribution_share = attribution_share;
        this._attribution_model = attribution_model;
        this._attr_window = attr_window;
        this._attribution_value = attribution_value;
        this._attribution_source = attribution_source;
        this._touchpoint_type = touchpoint_type;
        this._touchpoint_ts = touchpoint_ts;
        this._attribution_method = attribution_method;
        this._decline_reason = decline_reason;
        this._auditing_token = auditing_token;
        this._linkage_key = linkage_key;
        this._attribution_setting = attribution_setting;
    }

    /**
     * Returns the scope of the attribution data.
     * Exmaple: 'click'.
     */
    get scope(): string {
        return this._scope;
    }

    /**
     * Set the scope of the attribution data.
     * @param {String} scope Touchpoint type
     */
    set scope(scope: string): void {
        this._scope = scope;
    }

    /**
     * Set the scope of the attribution data.
     * @param {String} scope Touchpoint type
     * @returns {AttributionData}
     */
    setScope(scope: string): AttributionData {
        this._scope = scope;
        return this;
    }

    /**
     * Returns the visit time of the attribution data.
     * Example: 1512345678900
     */
    get visit_time(): number {
        return this._visit_time;
    }

    /**
     * Set the visit time of the attribution data.
     * @param {Number} visit_time Time that the campaign_id or fbc was first received
     */
    set visit_time(visit_time: number): void {
        this._visit_time = visit_time;
    }

    /**
     * Set the visit time of the attribution data.
     * @param {Number} visit_time Time that the campaign_id or fbc was first received
     * @returns {AttributionData}
     */
    setVisitTime(visit_time: number): AttributionData {
        this._visit_time = visit_time;
        return this;
    }

    /**
     * Return the ad id of the attribution data
     * Example: '1233435554'
     */
    get ad_id(): string {
        return this._ad_id;
    }

    /**
     * Set the ad id of the attribution data.
     * @param {String} ad_id Meta-provided ad id from URL/deeplink
     */
    set ad_id(ad_id: string): void {
        this._ad_id = ad_id;
    }

    /**
     * Set the ad id of the attribution data.
     * @param {String} ad_id Meta-provided ad id from URL/deeplink
     * @returns {AttributionData}
     */
    setAdId(ad_id: string): AttributionData {
        this._ad_id = ad_id;
        return this;
    }

    /**
     * Return the adset id of the attribution data
     * Example: '1233435554'
     */
    get adset_id(): string {
        return this._adset_id;
    }

    /**
     * Set the adset id of the attribution data.
     * @param {String} adset_id Meta-provided adset id from URL/deeplink
     */
    set adset_id(adset_id: string): void {
        this._adset_id = adset_id;
    }

    /**
     * Set the adset id of the attribution data.
     * @param {String} adset_id Meta-provided adset id from URL/deeplink
     * @returns {AttributionData}
     */
    setAdsetId(adset_id: string): AttributionData {
        this._adset_id = adset_id;
        return this;
    }

    /**
     * Return the campaign id of the attribution data
     * Example: '1233435554'
     */
    get campaign_id(): string {
        return this._campaign_id;
    }

    /**
     * Set the campaign id of the attribution data.
     * @param {String} campaign_id Meta-provided campaign id from URL/deeplink
     */
    set campaign_id(campaign_id: string): void {
        this._campaign_id = campaign_id;
    }

    /**
     * Set the campaign id of the attribution data.
     * @param {String} campaign_id Meta-provided campaign id from URL/deeplink
     * @returns {AttributionData}
     */
    setCampaignId(campaign_id: string): AttributionData {
        this._campaign_id = campaign_id;
        return this;
    }

    /**
     * Returns the attribution share of the attribution data.
     * Example: 0.3
     */
    get attribution_share(): number {
        return this._attribution_share;
    }

    /**
     * Set the attribution share of the attribution data.
     * @param {Number} attribution_share Range [0-1] weight of credit assigned to the visit.
     */
    set attribution_share(attribution_share: number): void {
        this._attribution_share = attribution_share;
    }

    /**
     * Set the attribution share of the attribution data.
     * @param {Number} attribution_share Range [0-1] weight of credit assigned to the visit.
     * @returns {AttributionData}
     */
    setAttributionShare(attribution_share: number): AttributionData {
        this._attribution_share = attribution_share;
        return this;
    }

    /**
     * Return the attribution model of the attribution data
     * Example: 'last-click'
     */
    get attribution_model(): AttributionModel {
        return this._attribution_model;
    }

    /**
     * Set the attribution model of the attribution data.
     * @param {AttributionModel} attribution_model Advertiser attribution model
     */
    set attribution_model(attribution_model: AttributionModel): void {
        this._attribution_model = attribution_model;
    }

    /**
     * Set the attribution model of the attribution data.
     * @param {String} attribution_model Advertiser attribution model
     * @returns {AttributionData}
     */
    setAttributionModel(attribution_model: AttributionModel): AttributionData {
        this._attribution_model = attribution_model;
        return this;
    }

    /**
     * Return the attribution window of the attribution data
     * Example: 7
     */
    get attr_window(): number {
        return this._attr_window;
    }

    /**
     * Set the attribution window of the attribution data.
     * @param {Number} attr_window Attribution window in days
     */
    set attr_window(attr_window: number): void {
        this._attr_window = attr_window;
    }

    /**
     * Set the attribution window of the attribution data.
     * @param {Number} attr_window Attribution window in days
     * @returns {AttributionData}
     */
    setAttrWindow(attr_window: number): AttributionData {
        this._attr_window = attr_window;
        return this;
    }

    /**
     * Returns the attribution value of the attribution data.
     * Example: 3.45
     */
    get attribution_value(): number {
        return this._attribution_value;
    }

    /**
     * Set the attribution value of the attribution data.
     * @param {Number} attribution_value The share of value generated by this click-conversion pair that is attributed to Meta.
     */
    set attribution_value(attribution_value: number): void {
        this._attribution_value = attribution_value;
    }

    /**
     * Set the attribution value of the attribution data.
     * @param {Number} attribution_value The share of value generated by this click-conversion pair that is attributed to Meta.
     * @returns {AttributionData}
     */
    setAttributionValue(attribution_value: number): AttributionData {
        this._attribution_value = attribution_value;
        return this;
    }

    /**
     * Returns the attribution source of the attribution data.
     * Example: 'AMM'
     */
    get attribution_source(): string {
        return this._attribution_source;
    }

    /**
     * Set the attribution source of the attribution data.
     * @param {String} attribution_source The attribution source to differentiate the source of the data, e.g. whether this is from AMM or Custom Attribution or any other sources.
     */
    set attribution_source(attribution_source: string): void {
        this._attribution_source = attribution_source;
    }

    /**
     * Set the attribution source of the attribution data.
     * @param {String} attribution_source The attribution source to differentiate the source of the data, e.g. whether this is from AMM or Custom Attribution or any other sources.
     * @returns {AttributionData}
     */
    setAttributionSource(attribution_source: string): AttributionData {
        this._attribution_source = attribution_source;
        return this;
    }

    /**
     * Returns the touchpoint type of the attribution data.
     * Example: 'click'
     */
    get touchpoint_type(): string {
        return this._touchpoint_type;
    }

    /**
     * Set the touchpoint type of the attribution data.
     * @param {String} touchpoint_type The engagement type that caused the original credited conversion.
     */
    set touchpoint_type(touchpoint_type: string): void {
        this._touchpoint_type = touchpoint_type;
    }

    /**
     * Set the touchpoint type of the attribution data.
     * @param {String} touchpoint_type The engagement type that caused the original credited conversion.
     * @returns {AttributionData}
     */
    setTouchpointType(touchpoint_type: string): AttributionData {
        this._touchpoint_type = touchpoint_type;
        return this;
    }

    /**
     * Returns the touchpoint timestamp of the attribution data.
     * Example: 1512345678900
     */
    get touchpoint_ts(): number {
        return this._touchpoint_ts;
    }

    /**
     * Set the touchpoint timestamp of the attribution data.
     * @param {Number} touchpoint_ts The time when the touchpoint event occurred with the ad that the install was credited to.
     */
    set touchpoint_ts(touchpoint_ts: number): void {
        this._touchpoint_ts = touchpoint_ts;
    }

    /**
     * Set the touchpoint timestamp of the attribution data.
     * @param {Number} touchpoint_ts The time when the touchpoint event occurred with the ad that the install was credited to.
     * @returns {AttributionData}
     */
    setTouchpointTs(touchpoint_ts: number): AttributionData {
        this._touchpoint_ts = touchpoint_ts;
        return this;
    }

    /**
     * Returns the attribution method of the attribution data.
     * Example: 'ard'
     */
    get attribution_method(): AttributionMethod {
        return this._attribution_method;
    }

    /**
     * Set the attribution method of the attribution data.
     * @param {AttributionMethod} attribution_method The attribution method used to attribute the event.
     */
    set attribution_method(attribution_method: AttributionMethod): void {
        this._attribution_method = attribution_method;
    }

    /**
     * Set the attribution method of the attribution data.
     * @param {AttributionMethod} attribution_method The attribution method used to attribute the event.
     * @returns {AttributionData}
     */
    setAttributionMethod(attribution_method: AttributionMethod): AttributionData {
        this._attribution_method = attribution_method;
        return this;
    }

    /**
     * Returns the decline reason of the attribution data.
     * Example: 'attribute_to_other_source'
     */
    get decline_reason(): DeclineReason {
        return this._decline_reason;
    }

    /**
     * Set the decline reason of the attribution data.
     * @param {DeclineReason} decline_reason The decline reason for the attribution.
     */
    set decline_reason(decline_reason: DeclineReason): void {
        this._decline_reason = decline_reason;
    }

    /**
     * Set the decline reason of the attribution data.
     * @param {DeclineReason} decline_reason The decline reason for the attribution.
     * @returns {AttributionData}
     */
    setDeclineReason(decline_reason: DeclineReason): AttributionData {
        this._decline_reason = decline_reason;
        return this;
    }

    /**
     * Returns the auditing token of the attribution data.
     * Example: 'token123'
     */
    get auditing_token(): string {
        return this._auditing_token;
    }

    /**
     * Set the auditing token of the attribution data.
     * @param {String} auditing_token The auditing token for the attribution.
     */
    set auditing_token(auditing_token: string): void {
        this._auditing_token = auditing_token;
    }

    /**
     * Set the auditing token of the attribution data.
     * @param {String} auditing_token The auditing token for the attribution.
     * @returns {AttributionData}
     */
    setAuditingToken(auditing_token: string): AttributionData {
        this._auditing_token = auditing_token;
        return this;
    }

    /**
     * Returns the linkage key of the attribution data.
     * Example: 'key123'
     */
    get linkage_key(): string {
        return this._linkage_key;
    }

    /**
     * Set the linkage key of the attribution data.
     * @param {String} linkage_key The linkage key for the attribution.
     */
    set linkage_key(linkage_key: string): void {
        this._linkage_key = linkage_key;
    }

    /**
     * Set the linkage key of the attribution data.
     * @param {String} linkage_key The linkage key for the attribution.
     * @returns {AttributionData}
     */
    setLinkageKey(linkage_key: string): AttributionData {
        this._linkage_key = linkage_key;
        return this;
    }

    /**
     * Returns the attribution setting of the attribution data.
     */
    get attribution_setting(): AttributionSetting {
        return this._attribution_setting;
    }

    /**
     * Set the attribution setting of the attribution data.
     * @param {AttributionSetting} attribution_setting The attribution setting configuration.
     */
    set attribution_setting(attribution_setting: AttributionSetting): void {
        this._attribution_setting = attribution_setting;
    }

    /**
     * Set the attribution setting of the attribution data.
     * @param {AttributionSetting} attribution_setting The attribution setting configuration.
     * @returns {AttributionData}
     */
    setAttributionSetting(attribution_setting: AttributionSetting): AttributionData {
        this._attribution_setting = attribution_setting;
        return this;
    }


    /**
	 * Returns the normalized payload for the attribution data.
	 * @returns {Object} normalized attribution data payload.
	 */
	normalize(): Object {

		const attributionData = {};

		if (this.scope) {
			attributionData.scope = this.scope;
		}

		if (this.visit_time) {
			attributionData.visit_time = this.visit_time;
		}

        if (this.ad_id) {
			attributionData.ad_id = this.ad_id;
		}

		if (this.adset_id) {
			attributionData.adset_id = this.adset_id;
		}

        if (this.campaign_id) {
			attributionData.campaign_id = this.campaign_id;
		}

		if (this.attribution_share) {
			attributionData.attribution_share = this.attribution_share;
		}

        if (this.attribution_model) {
			attributionData.attribution_model = this.attribution_model;
		}

        if (this.attr_window) {
			attributionData.attr_window = this.attr_window;
        }

        if (this.attribution_value) {
            attributionData.attribution_value = this.attribution_value;
        }

        if (this.attribution_source) {
            attributionData.attribution_source = this.attribution_source;
        }

        if (this.touchpoint_type) {
            attributionData.touchpoint_type = this.touchpoint_type;
        }

        if (this.touchpoint_ts) {
            attributionData.touchpoint_ts = this.touchpoint_ts;
        }

        if (this.attribution_method) {
            attributionData.attribution_method = this.attribution_method;
        }

        if (this.decline_reason) {
            attributionData.decline_reason = this.decline_reason;
        }

        if (this.auditing_token) {
            attributionData.auditing_token = this.auditing_token;
        }

        if (this.linkage_key) {
            attributionData.linkage_key = this.linkage_key;
        }

        if (this.attribution_setting) {
            attributionData.attribution_setting = this.attribution_setting.normalize();
        }

		return attributionData;
	}
}
